/* ---------------------------------------------------------------
  wmo2001_map.c
     Create (lon, lat) text for WMO-2001 observation map
     Create (lon, lat) text of FIR nad G-AREA
     Calucrate a circle of the radar obs range (lon,lat) 
     
  Compile: gcc -lm wmo2001_map.c -o wmo2001_map
  Usage:   radar_range
  Output files:  text file for drawmark.gs & drawline.gs
                 "wmo_site"     : radar & ship sites
                 "wmo_point"    : flight area points
                 "wmo_area_all" : flight area (FIR & G-area)
                 "wmo_g150"     : 150 nm circle from Komatsu
 
                                       Dec 19, 2000   by Shin Satoh
 ------------------------------------------------------------------- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define   PI    3.141592653589
#define   RAD   (PI/180.0)
#define   A     6378136.0
#define   F     (1.0/298.257)
#define	  E2    (2.0*F-F*F)

int main(int argc, char *argv[])
{
    FILE *fp;
    static char  fout[64],cname[12],ccomm[18];
    static char  pn[20][12],pc[20][18],an[20][12];
    float   px[20],py[20],ax[20],ay[20];

    int     i,j,mark,iflag,num_az;
    float   cent_x,cent_y,x0,y0,x1,y1;
    float   x,y,az,start_ang,end_ang,azint,range;
    double  r_lat,l1,l4;
    double  lon,lat;

/* ------- site locations------- */
    site_location(pn,px,py,pc,an,ax,ay);

    strcpy(fout,"wmo_site");
    if((fp = fopen(fout,"w" )) == NULL ){
        printf("File cannot open ! : %s \n",fout); exit(1);   }
    printf(" ===== Output file: %s =====\n",fout);

    for (j=0; j<=16; j++){
      lon=px[j];
      lat=py[j];
      strcpy(cname,&pn[j][0]);
      strcpy(ccomm,&pc[j][0]);
      /* if (j<5) mark=1;  GrADS crosshair (Airport) */
      if (j<5) mark=3;  /* GrADS closed-circle (Airport) */
      if (j>4) mark=5;  /* GrADS closed-square (Radar, Ship) */
      printf(" %8.4f  %8.4f  %d  %s  %s\n",lon,lat,mark,cname,ccomm);
      fprintf(fp,"%10.6f  %10.6f  %d  %s  %s\n",lon,lat,mark,cname,ccomm);
    }
    close (fp);

/* ------- area points------- */
    strcpy(fout,"wmo_point");
    if((fp = fopen(fout,"w" )) == NULL ){
        printf("File cannot open ! : %s \n",fout); exit(1);   }
    printf(" ===== Output file: %s =====\n",fout);

    for (j=2; j<=19; j++){
      if(j != 7){
        lon=ax[j];
        lat=ay[j];
        strcpy(cname,&an[j][0]);
        mark=3;  /* GrADS closed-circle (area point) */
        printf(" %8.4f  %8.4f  %d  %s\n",lon,lat,mark,cname);
        fprintf(fp,"%10.6f  %10.6f  %d  %s\n",lon,lat,mark,cname);
      }
    }
    close (fp);

/* ------- flight area (FIR & G-area) ------- */
    strcpy(fout,"wmo_area_all");
    if((fp = fopen(fout,"w" )) == NULL ){
        printf("File cannot open ! : %s \n",fout); exit(1);   }
    printf(" ===== Output file: %s =====\n",fout);

  /* [FIR TOKYO] */
    printf(" FIR TOKYO\n");
    for (i=0; i<6; i++){
      if(i==0) j=0;  if(i==1) j=1;  if(i==2) j=2;
      if(i==3) j=5;  if(i==4) j=6;  if(i==5) j=7;
      lon=ax[j];
      lat=ay[j];
      iflag=1;
      if (i==0) iflag=0;  /* start draw-line */
      fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);

      /* interporate on lat=43, 42, 41 */
      if (i==1){
        x0=ax[1];  y0=ay[1];  x1=ax[2];  y1=ay[2];
        lat=43.0; lon=x1+(lat-y1)*(x0-x1)/(y0-y1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
        lat=42.0; lon=x1+(lat-y1)*(x0-x1)/(y0-y1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
        lat=41.0; lon=x1+(lat-y1)*(x0-x1)/(y0-y1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
      }
      /* interporate on lon=132, 131, 130 */
      if (i==4){
        x0=ax[6];  y0=ay[6];  x1=ax[7];  y1=ay[7];
        lon=132.0; lat=y1+(lon-x1)*(y0-y1)/(x0-x1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
        lon=131.0; lat=y1+(lon-x1)*(y0-y1)/(x0-x1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
        lon=130.0; lat=y1+(lon-x1)*(y0-y1)/(x0-x1);
        fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
      }

    }
  /* [G-1] */
    printf(" G1-Area\n");
    /* for (i=0; i<12; i++){ */
    for (i=1; i<12; i++){
      if(i==0) j=3;   if(i==1) j=5;    if(i==2) j=8;
      if(i==3) j=17;  if(i==4) j=18;   if(i==5) j=19;
      if(i==6) j=16;  if(i==7) j=9;    if(i==8) j=15;
      if(i==9) j=12;  if(i==10) j=13;  if(i==11) j=3;
      lon=ax[j];
      lat=ay[j];
      iflag=1;
      if (i==0) iflag=0;  /* start draw-line */
      if (i==1) iflag=0;
      /* printf("%10.6f  %10.6f  %d\n",lon,lat,iflag); */
      fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
    }
  /* [G-2] */
    printf(" G2-Area\n");
    for (i=0; i<5; i++){
      if(i==0) j=10;  if(i==1) j=11;    if(i==2) j=12;
      if(i==3) j=13;  if(i==4) j=10;
      lon=ax[j];
      lat=ay[j];
      iflag=1;
      if (i==0) iflag=0;  /* start draw-line */
      /* printf("%10.6f  %10.6f  %d\n",lon,lat,iflag); */
      fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
    }
  /* [G-3] */
    printf(" G3-Area\n");
    for (i=0; i<4; i++){
      if(i==0) j=12;  if(i==1) j=14;    if(i==2) j=15;
      if(i==3) j=12;
      lon=ax[j];
      lat=ay[j];
      iflag=1;
      if (i==0) iflag=0;  /* start draw-line */
      /* printf("%10.6f  %10.6f  %d\n",lon,lat,iflag); */
      fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
    }
  /* [G-4] */
    printf(" G4-Area\n");
    for (i=0; i<5; i++){
      if(i==0) j=16;  if(i==1) j=17;    if(i==2) j=18;
      if(i==3) j=19;  if(i==4) j=16;
      lon=ax[j];
      lat=ay[j];
      iflag=1;
      if (i==0) iflag=0;  /* start draw-line */
      /* printf("%10.6f  %10.6f  %d\n",lon,lat,iflag); */
      fprintf(fp,"%10.6f  %10.6f  %d\n",lon,lat,iflag);
    }
    close (fp);


/* --- 150 nm circle from Komatsu --- */
    strcpy(fout,"wmo_g150");
    if((fp = fopen(fout,"w" )) == NULL ){
        printf("File cannot open ! : %s \n",fout); exit(1);   }
    printf(" ===== Output file: %s =====\n",fout);

      cent_x = px[3];    /* Komatsu airport */ 
      cent_y = py[3]; 
      num_az = 100;
      r_lat = cent_y * RAD;
      l1 = RAD*A*cos(r_lat)/sqrt(1-E2*sin(r_lat)*sin(r_lat));
      l4 = RAD*A*(1-E2)/pow((1-E2*sin(r_lat)*sin(r_lat)), 1.5);

      range=150.0 * 1.852;  /* [nautical mile] to [km] */
      printf(" Circle center: %8.4fE  %7.4fN  range: %5.1f km\n",cent_x,cent_y,range);
      start_ang = 278.0;
      end_ang   = 15.0+360.0;
      azint = (end_ang-start_ang)/num_az;
      for (j=0; j<=num_az; j++){
         az = azint*j + start_ang;
         x=range*sin(az*RAD);
         y=range*cos(az*RAD);
         lon = x*1000./l1 + cent_x;
         lat = y*1000./l4 + cent_y;
         iflag=1;
         if (j==0) iflag=0;
         fprintf(fp,"%10.6f %10.6f %1d\n",lon,lat,iflag);
       }

      /* [G-1a, G-1b, G-1c, G-1d] */
      for (i=0; i<3; i++){
	if(i==0) az=330.0-35.0;       
	if(i==1) az=330.0;       
	if(i==2) az=330.0+35.0;       
        x=range*sin(az*RAD);
        y=range*cos(az*RAD);
        lon = x*1000./l1 + cent_x;
        lat = y*1000./l4 + cent_y;
        fprintf(fp,"%10.6f %10.6f %1d\n",cent_x,cent_y,0);
        fprintf(fp,"%10.6f %10.6f %1d\n",lon,lat,1);
      }

      fclose(fp);

/* --- 60 km radar range circle of each radar --- */
    strcpy(fout,"wmo_radar");
    if((fp = fopen(fout,"w" )) == NULL ){
        printf("File cannot open ! : %s \n",fout); exit(1);   }
    printf(" ===== Output file: %s =====\n",fout);

    for (i=6; i<=12; i++){
      cent_x = px[i]; 
      cent_y = py[i];
      num_az = 180;
      r_lat = cent_y * RAD;
      l1 = RAD*A*cos(r_lat)/sqrt(1-E2*sin(r_lat)*sin(r_lat));
      l4 = RAD*A*(1-E2)/pow((1-E2*sin(r_lat)*sin(r_lat)), 1.5);

      range=60.0;  /* [km] */
      printf(" Radar: %8.4fE  %7.4fN  range: %5.1f km\n",cent_x,cent_y,range);
      start_ang = 0.0;
      end_ang   = 360.0;
      azint = (end_ang-start_ang)/num_az;
      for (j=0; j<=num_az; j++){
         az = azint*j + start_ang;
         x=range*sin(az*RAD);
         y=range*cos(az*RAD);
         lon = x*1000./l1 + cent_x;
         lat = y*1000./l4 + cent_y;
         iflag=1;
         if (j==0) iflag=0;
         fprintf(fp,"%10.6f %10.6f %1d\n",lon,lat,iflag);
       }
    }
    fclose(fp);
    return(0);
}


int site_location(
   char pn[][12], float *px, float *py, char pc[][18],
   char an[][12], float *ax, float *ay)
{
      strcpy(&pn[0][0],"Nagoya");
      px[0]=136+55/60.0+30/3600.0;  py[0]=35+15/60.0+ 0/3600.0; strcpy(&pc[0][0],"Airport");
      strcpy(&pn[1][0],"Niigata");
      px[1]=139+ 7/60.0+ 0/3600.0;  py[1]=37+57/60.0+10/3600.0; strcpy(&pc[1][0],"Airport");
      strcpy(&pn[2][0],"Toyama");
      px[2]=137+11/60.0+30/3600.0;  py[2]=36+38/60.0+45/3600.0; strcpy(&pc[2][0],"Airport");
      strcpy(&pn[3][0],"Komatsu");
      px[3]=136+24/60.0+45/3600.0;  py[3]=36+23/60.0+30/3600.0; strcpy(&pc[3][0],"Airport");
      strcpy(&pn[4][0],"Yonago");
      px[4]=133+14/60.0+50/3600.0;  py[4]=35+29/60.0+45/3600.0; strcpy(&pc[4][0],"Airport");
      strcpy(&pn[5][0],"Nagaoka");
      px[5]=138+53/60.0+24/3600.0;  py[5]=37+25/60.0+23/3600.0; strcpy(&pc[5][0],"ILTS");
      strcpy(&pn[6][0],"Kashiwazaki");
      px[6]=138+33/60.0+48/3600.0;  py[6]=37+22/60.0+52/3600.0; strcpy(&pc[6][0],"ILTS");
      strcpy(&pn[7][0],"Joetsu");
      px[7]=138+13/60.0+22/3600.0;  py[7]=37+10/60.0+ 2/3600.0; strcpy(&pc[7][0],"NIED");
      strcpy(&pn[8][0],"Oumi");
      px[8]=137+49/60.0+55/3600.0;  py[8]=37+ 2/60.0+10/3600.0; strcpy(&pc[8][0],"NIED");
      strcpy(&pn[9][0],"Oshimizu");     
      px[9]=136+45/60.0+42/3600.0;  py[9]=36+49/60.0+ 4/3600.0; strcpy(&pc[9][0],"IHAS");
      strcpy(&pn[10][0],"Tedorigawa");  
      px[10]=136+29/60.0+16/3600.0; py[10]=36+27/60.0+58/3600.0; strcpy(&pc[10][0],"IHAS");
      strcpy(&pn[11][0],"Mikuni");
      px[11]=136+ 8/60.0+11/3600.0; py[11]=36+12/60.0+46/3600.0; strcpy(&pc[11][0],"DA");
      strcpy(&pn[12][0],"Ohama");
      px[12]=135+43/60.0+58/3600.0; py[12]=35+33/60.0+26/3600.0; strcpy(&pc[12][0],"1/24-2/15");
      strcpy(&pn[13][0],"Kofu");
      px[13]=138.0;  py[13]=40.0;  strcpy(&pc[13][0],"1/12-20, 1/24-31");
      strcpy(&pn[14][0],"Seifu");
      px[14]=136.0;  py[14]=38.5;  strcpy(&pc[14][0],"1/12-20, 1/24-2/1");
      strcpy(&pn[15][0],"Tyofu");
      px[15]=133.0;  py[15]=37.0;  strcpy(&pc[15][0],"1/12-22");
      strcpy(&pn[16][0],"Vladivostok");
      px[16]=132.0;  py[16]=43.2;  strcpy(&pc[16][0],"");

      strcpy(&an[0][0],"142FIR");   ax[0]=142.00;   ay[0]=45.70;
      strcpy(&an[1][0],"140FIR");   ax[1]=140.00;   ay[1]=45.70;
      strcpy(&an[2][0],"136FIR");   ax[2]=136.00;   ay[2]=40.50;
      strcpy(&an[3][0],"FIR40");    ax[3]=135.40;   ay[3]=40.00;
      strcpy(&an[4][0],"FIR39");    ax[4]=134.20;   ay[4]=39.00;
      strcpy(&an[5][0],"FIR38");    ax[5]=133.00;   ay[5]=38.00;
      strcpy(&an[6][0],"133FIR");   ax[6]=133.00;   ay[6]=37.50;
      strcpy(&an[7][0],"129FIR");   ax[7]=129.13;   ay[7]=34.67;
      strcpy(&an[8][0],"G1-W");     ax[8]=133.00;   ay[8]=37.00;
      strcpy(&an[9][0],"SADO");     ax[9]=138.00;   ay[9]=38.00;
      strcpy(&an[10][0],"G2-N");    ax[10]=136.00;  ay[10]=40.00;
      strcpy(&an[11][0],"G2-W");    ax[11]=136.00;  ay[11]=39.80;
      strcpy(&an[12][0],"G2-S");    ax[12]=137.00;  ay[12]=39.00;
      strcpy(&an[13][0],"G2-E");    ax[13]=137.00;  ay[13]=40.00;
      strcpy(&an[14][0],"G3-S");    ax[14]=138.00;  ay[14]=38.20;
      strcpy(&an[15][0],"G3-E");    ax[15]=138.00;  ay[15]=38.50;
      strcpy(&an[16][0],"G4-N");    ax[16]=136.50;  ay[16]=37.40;
      strcpy(&an[17][0],"G4-W");    ax[17]=134.00;  ay[17]=36.25;
      strcpy(&an[18][0],"G4-S");    ax[18]=135.40;  ay[18]=36.25;
      strcpy(&an[19][0],"G4-E");    ax[19]=136.50;  ay[19]=36.75;

      return(0);
}
