/**********************************************************
                                 Dec. 13, 2001  Shin Satoh
FUNCTION: int lh_prof()
SOURCE FILE: lh_prof.c
PURPOSE: Retrive the Latent Heating Profile

CALLING SEQUENCE:
    (I) float *rho         : air density in range bins [kg/m^3]
    (I) float *qvs         : saturated mixing ratio of vapar in range bins
    (I) float *flag_rain   : rain possible flag (NO_RAIN=0., POSSIBLE=1., CERTAIN=2.)
    (I) float *flag_rtype  : rain type flag (STRAT=0., CONV=1., OTHERS=2.)
    (I) float *flag_BB     : BB exist flag (NO=0., EXIST=1.)
    (I) float *sfcr        : surface rainfall rate [mm/hr]
    (I) float *sfcht       : surface height [km]
    (I) float *bbht        : bright band height [km]
    (I) float *cldt        : cloud top height [km]
    (I) float *cldb        : cloud bottom height [km]
    (I) float *trop        : tropopause height [km]
    (I) float qp[][49]     : mixing ratio of precipitation [kg/kg]
    (I) float vt[][49]     : terminal falling velocity of precipitation [m/s]
    (O) float w[][49       : estimated vertical wind (w) [m/s]
    (O) float fqp[][49]    : production rate of precipitaion [kg/s]
    (O) float lh[][49]     : calculated latent heating [K/hr]

RETURN: itaration number
************************************************************/
#include <math.h>
#include <stdio.h>

#define  AVG          1      /* averaging pixel number */
#define  YDIM        49      /* number of angle bins in output */
#define  ZDIM        80      /* number of vertical bins in output */
#define  ZRESO        0.25   /* vertical resolution in km */
#define  UNDEF    -999.9    /* missing data in output */

#define  SNOW_THRES  -1.8    /* threshold of vt0 to estimate snow region */
#define  LH_THRES     0.5    /* threshold of diff between srfR & integLH */
#define  ITMAX       99      /* maximum number of itaration */

int lh_prof (float *rho, float *qvs, float *flag_rain, float *flag_rtype, 
             float *flag_BB, float *sfcr, float *sfcht, 
             float *bbht, float *cldt, float *cldb, float *trop,
             float qp[][YDIM], float vt[][YDIM], float vt0[][YDIM], 
             float w[][YDIM], float fqp[][YDIM], float lh[][YDIM])
{
      int    i, j, k, it;
      int    itarN[YDIM], itarN_max=0;
      int    rainposs, raintype, rainBB, lhtype;
      float  a, b, c, z0, zkm;
      float  xqavg, xqavgn, wmax;
      float  adjc, adjs, adjsz;

      static float  clv=2.50e+6;      /* [J kg^-1] in 0 C */
      static float  clf=0.334e+6;     /* [J kg^-1] in 0 C */
      static float  cp=1004.0;        /* [J K^-1 kg^-1]  for dry air */
      static float  rhol=1000.0;      /* [kg m^-3] density of water */
      static float  dz=250.0;         /* [m] */
      float  xadvqp,xsqp,dqvsdz,pi0,ds,xevacon;

      float  lhsfcr, lhinteg, lhdataN;
      float  diflh, diflh0;

/* ------ angle bin loop ------ */
      for(j=0; j<YDIM; j++){
      for(i=0; i<AVG; i++){
        
	  /*--- rainposs: (0)no_rain, (1)possible, (2)ceratain ----
                raintype: (0)strat, (1)conv, (2)others
                rainBB:   (0)no_BB, (1)exist
              (int is used to avoid roundoff error of float)
	  --------------------------------------------------------*/
	  rainposs = (int)(flag_rain[j]+0.5);
          raintype = (int)(flag_rtype[j]+0.5);
          rainBB = (int)(flag_BB[j]+0.5);

          /*--- original latent heating type (lhtype) ------------- 
             (0)no_rain, (1)conv, (2)strat, (3)anvil, (4)shallow
	  --------------------------------------------------------*/
          lhtype=0;
          if(rainposs>=1){
             if(raintype==1) lhtype=1;
             if(raintype==0) lhtype=2;
             if(raintype==2){
	        if(rainBB==1) lhtype=3;     /* maybe nothing */
                if(rainBB==0 && cldt[j]>bbht[j]) lhtype=3;
                if(rainBB==0 && cldt[j]<=bbht[j]) lhtype=4;
             }
          }
        
/* ------ iteration loop ------ */
      if(lhtype>0) {                 /* Nothing to do in no_rain */  

        diflh=999.;   diflh0=999.; 
        adjc=0.0;   adjs=0.0;   adjsz=0.0;
        for(it=0; it<ITMAX; it++){
          if (fabs(diflh) > LH_THRES){

   /* --- estimate w-profile --- */
          /* z0 height [km] */
          if(lhtype==0) z0=0.0;                /* no rain */
          if(lhtype==1) z0=cldt[j]*-2.0;       /* conv */
          if(lhtype==2) z0=bbht[j]*0.001+3.0;  /* strat org:-0.5km*/
          if(lhtype==3) z0=cldb[j];            /* anvil */
          if(lhtype==4) z0=cldt[j]*-2.0;       /* shallow */
          if(lhtype==2){
             z0-=adjsz;
             if(z0<0.0){
                z0=0.0;
                diflh=0.0;
             }
          }

          /* vertical averaged qp and estimate wmax */
          xqavg=0.0;
          xqavgn=0.0;
	  for(k=0; k<ZDIM; k++){
             if(qp[k][j]>0.0){
                xqavg+=qp[k][j];
                xqavgn+=1.0;
             }
          }
          if(xqavgn>1.0){
            xqavg/=xqavgn;
          }
          if (lhtype==0) wmax=0.0;                 /* no rain */
          if (lhtype==1) wmax=xqavg*2.0e3+adjc;    /* conv */
          if (lhtype==2) wmax=xqavg*20.0e3+adjs;   /* strat */
          if (lhtype==3) wmax=xqavg*20.0e3+adjs;   /* anvil */
          if (lhtype==4) wmax=xqavg*2.0e3+adjc;    /* shallow */

          /* get coeficients of the w profile */
	  for(k=0; k<ZDIM; k++){
	    zkm=k*ZRESO - sfcht[j];
            if(wmax>0.0 && z0!=0.0){
               a = -1.0*wmax*0.001;
               b = -1.0 * a * (cldt[j]+z0);
	       c = a * z0 * cldt[j];
            }else{
	      a=0.0;  b=0.0;  c=0.0;
            }
            w[k][j]=a*pow(zkm,3.0) + b*pow(zkm,2.0) + c*zkm;
            if(zkm>cldt[j] || zkm<sfcht[j]) w[k][j]=0.0;
          }

   /* ---- calc LH-profile ---- */
          /* get Fqp */
          for(k=0; k<ZDIM-1; k++){
             if (qp[k+1][j]>0.0 && qp[k][j]>0.0){
	        xadvqp=w[k][j]*(qp[k+1][j]-qp[k][j])/dz;
                xsqp=(rho[k+1]*vt[k+1][j]*qp[k+1][j]
                     -rho[k]*vt[k][j]*qp[k][j])
	             /(dz*rho[k]);
                /* assume large horizontal advection in snow region */
	        if(vt0[k][j]>SNOW_THRES) xadvqp*=0.3;   
                fqp[k][j]=xadvqp+xsqp;
             }else{
                fqp[k][j]=UNDEF;
             }
          }

          /* get LH-profile */
          for(k=0; k<ZDIM-1; k++){
	     if(fqp[k][j]>UNDEF){
                dqvsdz=(qvs[k+1]-qvs[k])/dz;
                if(fqp[k][j] > 0.0){
	           ds=1.0;    /* saturated */
                }else{
	           ds=0.0;    /* un-saturated */
                }
                xevacon=-1.0*ds*w[k][j]*dqvsdz + (1.0-ds)*fqp[k][j];
                if(vt0[k][j]>SNOW_THRES){  /* snow region */
                   lh[k][j]=((clv+clf)/cp)*xevacon*3600.0;   /* [K/hr] */
                }else{
                   lh[k][j]=(clv/cp)*xevacon*3600.0;
                }
                /* lh[k][j]=(clv/cp)*xevacon*3600.0; /* not consider ice phase */
             }else{
	        lh[k][j]=UNDEF;
             }
          }


   /* ---- comparison between integ-LH and srfR-LH ----- */
          /* vertical integ. of lh */
          lhinteg=0.0;  lhdataN=0.0;
          for(k=0; k<ZDIM; k++){
            if(qp[k][j]>0.0){
	       if(fabs(lh[k][j])<99.0){
                  lhinteg+=rho[k]*(lh[k][j]/3600.)*250.0;
                  lhdataN +=1.0;
               }
            }
          }

	  /* compare LHinteg and sfcR */
          if(sfcr[j]>0.7){
             lhsfcr = 2.50e6/1004.0*sfcr[j]/3600.;
          }else{
             lhsfcr = 0.0;
          }
          if(lhdataN > 3.0){
             diflh=lhsfcr-lhinteg;
          }else{
	     diflh=0.0;
          }

          /* adjustment coefficients of w-profile */
          if(fabs(diflh) > LH_THRES){
	      if(lhtype==1 || lhtype==4){
                 adjc+=diflh*0.002;
                 if(adjc>3.0 && diflh0>diflh) adjc-=diflh*0.003; 
	      }
              if(lhtype==2 || lhtype==3){
                 adjs+=diflh*0.20;
                 adjsz+=diflh*0.05;
                 /* if(adjs>20.0) adjs-=diflh*0.20; */
              }
          }

          /* case of failure in the iteration */
          if(j==24 && it==ITMAX-1){
             printf("== Itar=%2d y=%2d type=%1d:  diflh=%f  diflh0=%f | ",
		     it,j,lhtype,diflh,diflh0);
             printf("adjc=%f adjs=%f adjsz=%f wmax=%f\n",
		     adjc,adjs,adjsz,wmax);
          }
          diflh0=diflh;


          } else {    /* if(fabs(diflh) > LH_THRES) */
            itarN[j]=it;
          }
	}  /* end of iteration loop */
      }    /* end of if(lhtype>0) */

      if(itarN_max < itarN[j]) itarN_max = itarN[j];

      }}  /* end of angle_bin loop */

       return itarN_max;
}
