/**********************************************************
                                 Dec. 13, 2001  Shin Satoh
FUNCTION: int wc_param()
SOURCE FILE: wc_param.c
PURPOSE: Calculate the coefficients in WC-Z relation

Original Code by T. Iguchi (zlpara_4.c in 2A25 Version 6.3)

CALLING SEQUENCE:
    (I) int   *rbinNum     : range bin numbers of top[0], bottom[1],
                             surface[2], BB[3], PIZ_threshold[4], ZeMax[5]
    (I) int   raintype     : rain type (NO=-1, STRAT=0, CONV=1, OTHERS=2)
    (I) float zl_a_c0[3][5]: a in LWC = a * Z^b
    (I) float zl_a_c1[3][5]: a = 10^(a_c0+a_c1*x+a_c2*x^2)
    (I) float zl_a_c2[3][5]:   where x = log10(epsilon)
    (I) float zl_b_c0[3][5]: b in LWC = a * Z^b
    (I) float zl_b_c1[3][5]: b = 10^(b_c0+b_c1*x+b_c2*x^2)
    (I) float zl_b_c2[3][5]:   where x = log10(epsilon)
    (I) float nubfCFzl     : non-uniform beam filling correction factor
    (I) float epsilonf     : attenuation correction factor
    (I) float wght_ZR_corr : weighting factor for ZR correction 
    (I) int   *rbinZRNode  : node range bin numbers of Z-R coefficients
    (O) float *a           : a in LWC = a*Z^b in 80 range bins
    (O) float *b           : b in LWC = a*Z^b in 80 range bins
RETURN: normal=0
        no rain=1

Note: The original "zl" means Z-LWC, but LWC includes cloud water
      content in general.  Naturally, since TRMM PR measures only 
      precipitaion, the term of water content of precipitation (WC)
      is prefered.  (personal opinion of S. Satoh)
************************************************************/
#include <math.h>
#include <stdio.h>

#define  N5         5      /* number of nodes for WC-Z relations */
#define  NRANGE    80      /* number of vertical bins */

#define LDSNOW 0
#define HDSNOW 1
#define BBPEAK 2
#define RAIN0C 3
#define RAIN20 4

#define BBTHICK 2
#define MLTHICK 3
#define LPRATE  0.075

int wc_param(int *rbinNum, int raintype, int flag_warm, float zl_a_c0[][N5],  
        float zl_a_c1[][N5], float zl_a_c2[][N5], float zl_b_c0[][N5],  
        float zl_b_c1[][N5], float zl_b_c2[][N5], float nubfCFzl,  
        float epsilonf, float wght_ZR_corr, int *rbinZRNode, 
        float *a, float *b)
{
    int    i, j, k, kp1;
    int    rbinTop, rbinBottom, rbinBB, rbin_mlpeak, rbin0C, dl;
    float  fraction;
    float  zl_a[N5], zl_b[N5];         /* a and b in R=a*Z^b */
    float  zlaNode[N5], zlbNode[N5];   /* actual a and b in the Node */
    float  log_epsi;             /* log10(epsilon)   */
    float  log_epsi_2;           /* log_epsi * log_epsi  */
    float  epsi_w;               /* weighted epsilon */

/* --- in case of no rain ---*/
    if(raintype < 0){
       for(j=0;j<NRANGE;j++) {
          a[j] = 0.0;
          b[j] = 1.0;
       }
       return 1;
    }

/* --- modifications of a and b according to epsilonf --- */
    if(wght_ZR_corr > 0.0) {
        epsi_w = 1.0 + wght_ZR_corr*(epsilonf-1.0);
        if     (epsi_w>2.0) log_epsi =  0.30103; /* 2.00 =  3.0103 dB */
        else if(epsi_w<0.5) log_epsi = -0.30103; /* 0.50 = -3.0103 dB */
        else {
            log_epsi = log10(epsi_w);
        }
        log_epsi_2 = log_epsi * log_epsi;
        for(i=0;i<N5;i++) {
            zl_a[i] = zl_a_c0[raintype][i] 
                    + zl_a_c1[raintype][i]*log_epsi 
                    + zl_a_c2[raintype][i]*log_epsi_2 ;
            zl_b[i] = zl_b_c0[raintype][i] 
                    + zl_b_c1[raintype][i]*log_epsi 
                    + zl_b_c2[raintype][i]*log_epsi_2 ;
            zl_a[i] = pow(10.0,zl_a[i]);
            zl_b[i] = pow(10.0,zl_b[i]);
        }
    } else {
        for(i=0;i<N5;i++) {
            zl_a[i] = pow(10.0,zl_a_c0[raintype][i]);
            zl_b[i] = pow(10.0,zl_b_c0[raintype][i]);
        }
    }


/* --- modifications of a and b using rbinNum --- */
    rbinTop = rbinNum[0];
    rbinBottom = rbinNum[1];
    rbinBB = rbinNum[3];
    rbin0C = rbinNum[3];   /* <==== NO DATA in 2A25 */

    /* --- 0C between rain top and bottom --- */
    if(rbinBB < NRANGE ||                         /* if BB exists */
        (rbin0C>=rbinTop && rbin0C<=rbinBottom)) { /* if 0C between rain top and bottom */

        if(rbinBB < NRANGE) {  /* if BB exists */
            dl = BBTHICK;
            rbin_mlpeak = rbinBB;
            rbinZRNode[1] = rbin_mlpeak-dl;
            if(rbinZRNode[1]<0) rbinZRNode[1]=0;
            rbinZRNode[2] = rbin_mlpeak;
            rbinZRNode[3] = rbin_mlpeak+dl;
            if(rbinZRNode[3]>NRANGE) rbinZRNode[3]=NRANGE;
            rbinZRNode[4] = rbinBottom;
            if(rbinZRNode[3]>rbinBottom) {
                rbinZRNode[3] = rbinBottom;
                zlaNode[3] = zl_a[BBPEAK]+ (rbinZRNode[3]-rbin_mlpeak)*
                            (zl_a[RAIN0C]-zl_a[BBPEAK])/(float)dl;
                zlaNode[4] = zlaNode[3];
                zlbNode[3] = zl_b[BBPEAK]+ (rbinZRNode[3]-rbin_mlpeak)*
                            (zl_b[RAIN0C]-zl_b[BBPEAK])/(float)dl;
                zlbNode[4] = zlbNode[3];
            } else {
                zlaNode[3] = zl_a[RAIN0C];
                zlaNode[4] = zl_a[RAIN0C]+ (rbinZRNode[4]-rbin_mlpeak)*
                            LPRATE*(zl_a[RAIN20]-zl_a[RAIN0C]);
                zlbNode[3] = zl_b[RAIN0C];
                zlbNode[4] = zl_b[RAIN0C]+ (rbinZRNode[4]-rbin_mlpeak)*
                            LPRATE*(zl_b[RAIN20]-zl_b[RAIN0C]);
            } 
            zlaNode[0] = zl_a[LDSNOW];
            zlaNode[1] = zl_a[HDSNOW];
            zlaNode[2] = zl_a[BBPEAK];
            zlbNode[0] = zl_b[LDSNOW];
            zlbNode[1] = zl_b[HDSNOW];
            zlbNode[2] = zl_b[BBPEAK];
        } else { 
            dl = MLTHICK;
            rbin_mlpeak = rbin0C;
            rbinZRNode[1] = rbin_mlpeak-dl;
            if(rbinZRNode[1]<0) rbinZRNode[1]=0;
            rbinZRNode[2] = rbin_mlpeak;
            rbinZRNode[3] = rbin_mlpeak+dl;
            if(rbinZRNode[3]>NRANGE) rbinZRNode[3]=NRANGE;
            rbinZRNode[4] = rbinBottom;
            if(rbinZRNode[3]>rbinBottom) {
                rbinZRNode[3] = rbinBottom;
                zlaNode[3] = zl_a[RAIN0C];
                zlaNode[4] = zlaNode[3];
                zlbNode[3] = zl_b[RAIN0C];
                zlbNode[4] = zlbNode[3];
            } else {
                zlaNode[3] = zl_a[RAIN0C];
                zlaNode[4] = zl_a[RAIN0C]+ (rbinZRNode[4]-rbinZRNode[3])*
                            LPRATE*(zl_a[RAIN20]-zl_a[RAIN0C]);
                zlbNode[3] = zl_b[RAIN0C];
                zlbNode[4] = zl_b[RAIN0C]+ (rbinZRNode[4]-rbinZRNode[3])*
                            LPRATE*(zl_b[RAIN20]-zl_b[RAIN0C]);
            } 
            zlaNode[0] = zl_a[LDSNOW];
            zlaNode[1] = zl_a[HDSNOW];
            zlaNode[2] = zl_a[RAIN0C];
            zlbNode[0] = zl_b[LDSNOW];
            zlbNode[1] = zl_b[HDSNOW];
            zlbNode[2] = zl_b[RAIN0C];
        }
    } else {
        rbinZRNode[1] = rbinBottom;
        /* -- 0C above the rain top -- */
        if(rbin0C<rbinTop || flag_warm>0 ) {
            zlaNode[0] = zl_a[RAIN0C]+ (rbinZRNode[0]-rbin0C)*
                            LPRATE*(zl_a[RAIN20]-zl_a[RAIN0C]);
            zlaNode[1] = zl_a[RAIN0C]+ (rbinZRNode[1]-rbin0C)*
                            LPRATE*(zl_a[RAIN20]-zl_a[RAIN0C]);
            zlbNode[0] = zl_b[RAIN0C]+ (rbinZRNode[0]-rbin0C)*
                            LPRATE*(zl_b[RAIN20]-zl_b[RAIN0C]);
            zlbNode[1] = zl_b[RAIN0C]+ (rbinZRNode[1]-rbin0C)*
                            LPRATE*(zl_b[RAIN20]-zl_b[RAIN0C]);
        }
        /* -- 0C below the bottom -- */
        else if (rbin0C > rbinBottom) {
            zlaNode[0] = zl_a[LDSNOW];
            zlbNode[0] = zl_b[LDSNOW];
            zlaNode[1] = zl_a[HDSNOW];
            zlbNode[1] = zl_b[HDSNOW];
        }
        /* -- if 0C height is not available  -- */
        else if(rbin0C == 0) {
            zlaNode[0] = zl_a[RAIN0C];
            zlbNode[0] = zl_b[RAIN0C];
            zlaNode[1] = zl_a[RAIN20];
            zlbNode[1] = zl_b[RAIN20];
        }
        /* this is an impossible path, but added to avoid accidental error */
        else {  
             zlaNode[0] = zl_a[RAIN0C];
             zlbNode[0] = zl_b[RAIN0C];
             zlaNode[1] = zl_a[RAIN20];
             zlbNode[1] = zl_b[RAIN20];
        }
        for(j=2;j<5;j++) {
            rbinZRNode[j] = rbinBottom;
            zlaNode[j] = zlaNode[1];
            zlbNode[j] = zlbNode[1];
        }
    }

/* --- get a and b in all range bins (NRANGE) ---*/
    for(j=0;j<rbinZRNode[0];j++) {
        a[j] = 0.0;
        b[j] = 1.0;
    }
    for(k=0;k<4;k++) {
        kp1 = k+1;
        for(j=rbinZRNode[k];j<rbinZRNode[kp1];j++) {
            fraction = (float)(j-rbinZRNode[k])/
                (float)(rbinZRNode[kp1]-rbinZRNode[k]);
            a[j] = zlaNode[k] + (zlaNode[kp1]-zlaNode[k])*fraction;
            b[j] = zlbNode[k] + (zlbNode[kp1]-zlbNode[k])*fraction;
        }
    }
    a[rbinZRNode[4]]= zlaNode[4];
    b[rbinZRNode[4]]= zlbNode[4];

    for(j=rbinZRNode[4]+1;j<NRANGE;j++) {
        a[j] = 0.0;
        b[j] = 1.0;
    }

/* --- non-uniform beam filling correction --- */
    for(j=rbinZRNode[0];j<=rbinZRNode[4];j++) {
        a[j] *= nubfCFzl;
    }

#ifdef DEBUG0
    printf("### zlaNode[0]=%10.4f  [1]=%10.4f  [2]=%10.4f  [3]=%10.4f  [4]=%10.4f\n",
           zlaNode[0],zlaNode[1],zlaNode[2],zlaNode[3],zlaNode[4]);
    printf("### zlbNode[0]=%10.4f  [1]=%10.4f  [2]=%10.4f  [3]=%10.4f  [4]=%10.4f\n",
           zlbNode[0],zlbNode[1],zlbNode[2],zlbNode[3],zlbNode[4]);
    for(k=rbinTop; k<rbinBottom; k++){
        printf("### k=%2d  zwc_a=%10.4f  zwc_b=%10.4f\n",k,a[k],b[k]);
    }
#endif

    return 0;
}
