/**********************************************************
                                 Dec. 13, 2001  Shin Satoh
FUNCTION: int zwc_param()
SOURCE FILE: zwc_param.c
PURPOSE: Calculate the coefficients in WC-Z relation

Original Code by T. Iguchi (zlpara_4.c in 2A25 Version 6.3)

CALLING SEQUENCE:
    (I) int   *rbinNum     : range bin numbers of top[0], bottom[1],
                             surface[2], BB[3], PIZ_threshold[4], ZeMax[5]
    (I) int   raintype     : rain type (NO=-1, STRAT=0, CONV=1, OTHERS=2)
    (I) float zl_a_c0[3][5]: a in LWC = a * Z^b
    (I) float zl_a_c1[3][5]: a = 10^(a_c0+a_c1*x+a_c2*x^2)
    (I) float zl_a_c2[3][5]:   where x = log10(epsilon)
    (I) float zl_b_c0[3][5]: b in LWC = a * Z^b
    (I) float zl_b_c1[3][5]: b = 10^(b_c0+b_c1*x+b_c2*x^2)
    (I) float zl_b_c2[3][5]:   where x = log10(epsilon)
    (I) float nubfCFzl     : non-uniform beam filling correction factor
    (I) float epsilonf     : attenuation correction factor
    (I) float wght_ZR_corr : weighting factor for ZR correction 
    (I) int   *rbinZRNode  : node range bin numbers of Z-R coefficients
    (O) float *a           : a in LWC = a*Z^b in 80 range bins
    (O) float *b           : b in LWC = a*Z^b in 80 range bins
RETURN: normal=0
        no rain=1

Note: The original "zl" means Z-LWC, but LWC includes cloud water
      content in general.  Naturally, since TRMM PR measures only 
      precipitaion, the term of water content of precipitation (WC)
      is prefered.  (personal opinion of S. Satoh)
************************************************************/
#include <math.h>
#include <stdio.h>

#define  N5         5      /* number of nodes for WC-Z relations */
#define  N6         6      /* number of range bin numbers */
#define  NRANGE    80      /* number of vertical bins */

int zwc_param(int *rbinNum, int raintype, float zl_a_c0[][N5],  
        float zl_a_c1[][N5], float zl_a_c2[][N5], float zl_b_c0[][N5],  
        float zl_b_c1[][N5], float zl_b_c2[][N5], float nubfCFzl,  
        float epsilonf, float wght_ZR_corr, int *rbinZRNode, 
        float *a, float *b)
{
    int    i,j,k,kp1;
    float  fraction;
    float  zl_a[N5], zl_b[N5];   /* a and b in R=a*Z^b */
    float  log_epsi;             /* log10(epsilon)   */
    float  log_epsi_2;           /* log_epsi * log_epsi  */
    float  epsi_w;               /* weighted epsilon */


/* --- in case of no rain ---*/
    if(raintype < 0){
       for(j=0;j<NRANGE;j++) {
          a[j] = 0.0;
          b[j] = 1.0;
       }
       return 1;
    }

/* --- modifications of a and b according to epsilonf --- */

    if(wght_ZR_corr > 0.0) {
        epsi_w = 1.0 + wght_ZR_corr*(epsilonf-1.0);
        if     (epsi_w>2.0) log_epsi =  0.30103; /* 2.00 =  3.0103 dB */
        else if(epsi_w<0.5) log_epsi = -0.30103; /* 0.50 = -3.0103 dB */
        else {
            log_epsi = log10(epsi_w);
        }
        log_epsi_2 = log_epsi * log_epsi;
        for(i=0;i<N5;i++) {
            zl_a[i] = zl_a_c0[raintype][i] 
                    + zl_a_c1[raintype][i]*log_epsi 
                    + zl_a_c2[raintype][i]*log_epsi_2 ;
            zl_b[i] = zl_b_c0[raintype][i] 
                    + zl_b_c1[raintype][i]*log_epsi 
                    + zl_b_c2[raintype][i]*log_epsi_2 ;
            zl_a[i] = pow(10.0,zl_a[i]);
            zl_b[i] = pow(10.0,zl_b[i]);
        }
    } else {
        for(i=0;i<N5;i++) {
            zl_a[i] = pow(10.0,zl_a_c0[raintype][i]);
            zl_b[i] = pow(10.0,zl_b_c0[raintype][i]);
        }
    }

    for(j=0;j<rbinZRNode[0];j++) {
        a[j] = 0.0;
        b[j] = 1.0;
    }
    for(k=0;k<4;k++) {
        kp1 = k+1;
        for(j=rbinZRNode[k];j<rbinZRNode[kp1];j++) {
            fraction = (float)(j-rbinZRNode[k])/
                (float)(rbinZRNode[kp1]-rbinZRNode[k]);
            a[j] = zl_a[k] + (zl_a[kp1]-zl_a[k])*fraction;
            b[j] = zl_b[k] + (zl_b[kp1]-zl_b[k])*fraction;
        }
    }
    a[rbinZRNode[4]]= zl_a[4];
    b[rbinZRNode[4]]= zl_b[4];

    for(j=rbinZRNode[4]+1;j<NRANGE;j++) {
        a[j] = 0.0;
        b[j] = 1.0;
    }

/*------------------------------------------------------------
    non-uniform beam filling correction
------------------------------------------------------------*/
    for(j=rbinZRNode[0];j<=rbinZRNode[4];j++) {
        a[j] *= nubfCFzl;
    }
    return 0;
}


