/**********************************************************
                                 June 21, 2002  Shin Satoh
FUNCTION: int lh_prof()
SOURCE FILE: lh_prof.c
PURPOSE: Retrive the Latent Heating Profile

CALLING SEQUENCE:
    (I) float *rho         : air density in range bins [kg/m^3]
    (I) float *qvs         : saturated mixing ratio of vapar in range bins
    (I) float *flag_rain   : rain possible flag (NO_RAIN=0., POSSIBLE=1., CERTAIN=2.)
    (I) float *flag_rtype  : rain type flag (STRAT=0., CONV=1., OTHERS=2.)
    (I) float *flag_BB     : BB exist flag (NO=0., EXIST=1.)
    (I) float *sfcr        : near surface rainfall rate [mm/hr]
    (I) float *sfcht       : surface height [km]
    (I) float *bbht        : bright band height [km]
    (I) float *cldt        : cloud top height [km]
    (I) float *cldb        : cloud bottom height [km]
    (I) float *trop        : tropopause height [km]
    (I) float qp[][49]     : mixing ratio of precipitation [kg/kg]
    (I) float vt[][49]     : terminal falling velocity of precipitation [m/s]
    (O) float w[][49       : estimated vertical wind (w) [m/s]
    (O) float fqp[][49]    : production rate of precipitaion [kg/s]
    (O) float lh[][49]     : calculated latent heating [K/hr]

DEBUG:
  DEBUG1:  iteration
  DEBUG2:  profiles of some variables

RETURN: max itaration number, but 99 means 1st guess error
************************************************************/
#include <math.h>
#include <stdio.h>

#define  YDIM        49      /* number of angle bins in output */
#define  ZDIM        80      /* number of vertical bins in output */
#define  ZRESO        0.25   /* vertical resolution in km */
#define  UNDEF    -999.9    /* missing data in output */

#define  SNOW_THRES  -1.8    /* threshold of vt0 to estimate snow region */
#define  LH_THRES     0.5    /* threshold of diff between srfR & integLH */
#define  WMAX_LIMIT  10.0    /* limitation of w maximum */
#define  ITMAX       30      /* maximum number of itaration */

int lh_prof (float *rho, float *qvs, float *flag_wtype, float *sfcr,  
             float *sfcht, float *bbht, float *cldt, float *cldb, float *trop,
             float qp[][YDIM], float vt[][YDIM], float vt0[][YDIM], 
             float w[][YDIM], float fqp[][YDIM], float lh[][YDIM])
{
      int    i, j, k, it, it2;
      int    itarN[YDIM], itarN_max=0;
      int    wtype, lowest_flag, status;
      float  xqavg, xqavgn, xvt0, wmax, pmotion;
      float  advqp, advqph, srcqp, dqvsdz, xevacon;
      double zkm, zt, zc, zm;

      float  LHsfcR, LHinteg, LHintegS, LHintegN;
      float  wtmp, guessR, guessLH, satLH;
      float  adjw, adjw0, adjw1, adjadv;
      float  difLH, difLH0, difLH1;

      static float  coef[5], coef0[5];   /* coefficients of w-prof */
      static float  clv=2.50e+6;         /* [J kg^-1] in 0 C */
      static float  clf=0.334e+6;        /* [J kg^-1] in 0 C */
      static float  cp=1004.0;           /* [J K^-1 kg^-1]  for dry air */
      static float  rhol=1000.0;         /* [kg m^-3] density of water */
      static float  dzm=ZRESO*1000.0;    /* [m] */


/*======================================================================*/
/* ------ init output values ------ */
      for(j=0; j<YDIM; j++){
        for(k=0; k<ZDIM; k++){
          w[k][j]=UNDEF;
          fqp[k][j]=UNDEF;
	  lh[k][j]=UNDEF;    
        }
      }

/* ------ angle bin loop ------ */
      for(j=0; j<YDIM; j++){
        
      itarN[j]=0;
      /*--- w-profile type (flag_wtype) ---------------------- 
      (0)no_rain, (1)conv, (2)strat, (3)anvil, (4)shallow
       --------------------------------------------------------*/
      wtype = (int)(flag_wtype[j]+0.5);

      if(wtype>0){     /* Nothing to do in no_rain */

 /* --- get initial coefficients of w-profile ---
        zt, zc, zm are heights from the surface --- */
        zt = trop[j]-sfcht[j];
        zc = cldt[j]-sfcht[j];
        if(wtype==1) zm = 0.01;                     /* conv */
        /* if(wtype==2) zm = bbht[j]-0.5-sfcht[j];     strat */
        if(wtype==2) zm = bbht[j]-sfcht[j];         /* strat */
        if(wtype==3) zm = cldb[j]+0.25-sfcht[j];    /* anvil */
        if(wtype==4) zm = 0.01;                     /* shallow */

        /* Check the 3 nodes */
        if(zm<0.01) zm=0.01;
        if(zc<zm)   zc=zm+0.25;
        if(zt<zc)   zt=zc+0.25;

        /* Gauss-Jordan method to solve the 4th power func */
        status = w4pf_prof(zt, zc, zm, coef0);
        if(status!=0){
	   printf("## Error of 4th power function calc: status=%d\n",status);
           printf("## j=%d wtype=%d trop-sfc=%5.2f cldt-sfc=%5.2f melt/bot-sfc=%5.2f\n",
                      j,wtype,zt,zc,zm);
           printf("## a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f\n",
		      coef0[1],coef0[2],coef0[3],coef0[4]);   /* coef0[1]=1.0 */
           exit(1); 
        }

        /* Saturation LH integration for 1st guess -------------
         satLH = -(Lv/Cp) * w * (dqvs/dz)     : satLH --[K/s]
         Integ[ rho*LH ]dz = (Lv/Cp) * Rsfc   : Rsfc --[mm/s]
        -------------------------------------------------------*/
        LHintegS = 0.0;
        for(k=1; k<ZDIM-1; k++){
	  zkm = k*ZRESO - sfcht[j];           /* height [km] from the surface */
          dqvsdz = (qvs[k+1]-qvs[k-1])/(dzm*2.0);                  /* dqvs/dz */
          wtmp = pow(zkm,4.) + coef0[2]*pow(zkm,3.) +
	         coef0[3]*pow(zkm,2.) + coef0[4]*zkm;    /* temporary w [m/s] */ 
          xevacon=-1.0 * wtmp * dqvsdz;
          xvt0=vt0[k][j];
          if(xvt0>SNOW_THRES){  /* snow region */
             satLH=((clv+clf)/cp)*xevacon*3600.0;        /* [K/hr] */
          }else{
             satLH=(clv/cp)*xevacon*3600.0;
          }

          /* conv or shallow */
          if((wtype==1 || wtype==4) && 0.0<=zkm && zkm<=zc){
             LHintegS += rho[k] * satLH/3600.0 * dzm;
          }
          /* strat or anvil */
          if((wtype==2 || wtype==3) && zm<=zkm && zkm<=zc){
             LHintegS += rho[k] * satLH/3600.0 * dzm;
          }
        }
 
#ifdef DEBUG1
        printf("\n");
        printf("# j=%2d wtype=%d trop=%5.2f cldt=%5.2f melt=%5.2f sfcR=%5.1f\n",
                j,wtype,zt,zc,zm,sfcr[j]);
#endif

        /* make the first guess (smaller LHinteg) of the w-profile */
        if(sfcr[j]<0.5){          /* anvil or very weak sfcR */
	   guessR = 0.5;          /* [mm/hr] */
        }else{
	   guessR = sfcr[j];
        }
        guessLH = (clv/cp)*guessR/3600.; 
        adjw0=guessLH/LHintegS;     /* adjustment of w-profile max */

#ifdef DEBUG1
        if(adjw0<=0.0){
	  printf("### j=%2d  adjw0=%f  guessR=%f  LHintegS=%s\n",
		 j,adjw0,guessR,LHintegS);
        }
        coef[1]=adjw0;
        coef[2]=coef0[2]*coef[1];
        coef[3]=coef0[3]*coef[1];
        coef[4]=coef0[4]*coef[1];
        wmax = 0.0;
        for (k=0; k<ZDIM; k++) {
	  zkm = k*ZRESO - sfcht[j];
          w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	            coef[3]*pow(zkm,2.) + coef[4]*zkm;
          if(zkm < 0.0) w[k][j]=0.0;
          if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
          if(wmax < w[k][j]) wmax=w[k][j];
        }
        printf("# 1st guess: a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f  |  wmax=%6.2f\n",
                coef[1],coef[2],coef[3],coef[4],wmax);
#endif

        /* make the initial second guess (larger LHinteg ) of the w-profile */
        if(wtype==1){             /* conv */
          if(sfcr[j]<0.5){
            guessR=0.5*5.0;
          }else{
            guessR=sfcr[j]*5.0;
          }
	  if(guessR>300.0) guessR = 300.0;  /* [mm/hr] */
        }else{                    /* strat, anvil, shallow */  
          if(sfcr[j]<0.5){
            guessR=0.5*2.0;
          }else{
            guessR=sfcr[j]*2.0;
          }
	  if(guessR>100.0) guessR = 100.0;  /* [mm/hr] */
        }
        guessLH = (clv/cp)*guessR/3600.; 
        adjw1=guessLH/LHintegS;     /* adjustment of w-profile max */

#ifdef DEBUG1
        coef[1]=adjw1;
        coef[2]=coef0[2]*coef[1];
        coef[3]=coef0[3]*coef[1];
        coef[4]=coef0[4]*coef[1];
        wmax = 0.0;
        for (k=0; k<ZDIM; k++) {
	  zkm = k*ZRESO - sfcht[j];
          w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	            coef[3]*pow(zkm,2.) + coef[4]*zkm;
          if(zkm < 0.0) w[k][j]=0.0;
          if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
          if(wmax < w[k][j]) wmax=w[k][j];
        }
        printf("# 2nd guess: a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f  |  wmax=%6.2f\n",
                coef[1],coef[2],coef[3],coef[4],wmax);
#endif



/*=================================================================*/
/* ------ initialize for iteration ------ */
        difLH=99.9;   
        difLH0=99.9;
        difLH1=99.9;
        adjadv=0.0;    /* [-1.0-1.0] adjustment of horiz advection of qp  */
        it2=0;

/* ------ iteration loop ------ */
        for(it=0; it<ITMAX; it++){
          if (fabs(difLH) > LH_THRES){
	    if(it==0) adjw = adjw0;    /* 1st guess estimated on the above */
            if(it==1) adjw = adjw1;    /* 2nd guess estimated on the above */

   /* ---- calculate the w-profile ---- */
          coef[1]=adjw;
          coef[2]=coef0[2]*coef[1];
          coef[3]=coef0[3]*coef[1];
          coef[4]=coef0[4]*coef[1];
          wmax=0.0;
          for (k=0; k<ZDIM; k++) {
 	     zkm = k*ZRESO - sfcht[j];
             w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	               coef[3]*pow(zkm,2.) + coef[4]*zkm;
             if(zkm < 0.0) w[k][j]=0.0;
             if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
             if(wmax < w[k][j]) wmax=w[k][j];
          }
#ifdef DEBUG2
	  if(it<3){
            printf("#======== profiles at it=%2d(%2d): j=%2d wtype=%1d ",
	           it,it2,j,wtype);
            printf("sfcR=%7.3f : adjw=%f wmax=%f adjadv=%4.2f ==========\n",
	           sfcr[j],adjw,wmax,adjadv);
          }
#endif

   /* ---- calculate the LH-profile -----------------------------------------
         Fqp = advqph + w*dqp/dz + (1/rho)*d(rho*vt*qp)/dz
             (advqph = u*dqp/dx + v*dqp/dy : unknown term)
             (same Fqp = advqph + w*dqp/dz - (1/rho)*dR/dz)
         Fqp>0 && w>0: LH = -(Lv/Cp) * w * (dqvs/dz) * 3600 [K/hr]
         Fqp>0 && w<0: LH = 0
         Fqp<0 && w>0: LH = 0
         Fqp<0 && w<0: LH = Fqp * 3600                      [K/hr]
      -------------------------------------------------------------- */
          /* get Fqp and LH-profile [K/hr] */
          for(k=1; k<ZDIM-1; k++){
             if (qp[k+1][j]>0.0 && qp[k-1][j]>0.0){
	        /* vertical advection of qp */
	        advqp=w[k][j]*(qp[k+1][j]-qp[k-1][j])/(dzm*2.0);
                /* source of qp */
                srcqp=(1.0/rho[k])*(rho[k+1]*vt[k+1][j]*qp[k+1][j]
                                   -rho[k-1]*vt[k-1][j]*qp[k-1][j])/(dzm*2.0);
                /* assume unkwon horizontal advection of qp for adjustment */
                advqph = -1.0*advqp*adjadv;    /* adjadv: -1.0 to 1.0 */
                xvt0 = (vt0[k+1][j]+vt0[k-1][j])*0.5;
                if(xvt0>SNOW_THRES && adjadv==0.0){ 
                  advqph *= -1.0*advqp*0.7;
                }

                pmotion = w[k][j]+vt[k][j];
                /* w+vt0>0 & fqp<0 means positive advection */
                if(pmotion>0 && (advqp+srcqp)<0){
                  advqph = -1.01*(advqp+srcqp);
                }
                /* w<0 & fqp>0 means negative advection */
                if(w[k][j]<0 && (advqp+srcqp)>0){
                  advqph = -1.01*(advqp+srcqp);
                }

                fqp[k][j]=advqph + advqp + srcqp;

                /* LH-profile */
                dqvsdz=(qvs[k+1]-qvs[k-1])/(dzm*2.0);

                if(fqp[k][j] >= 0.0){   /* saturation */
                  xevacon=-1.0*w[k][j]*dqvsdz;
                }else{                  /* un-saturation */
		  xevacon=fqp[k][j];
                }

                /*--- old before V3.2 ------------
                if(fqp[k][j] > 0.0){   /* saturation
		  if(w[k][j]>0.0){     /* updraft
                    xevacon=-1.0*w[k][j]*dqvsdz;
                  }else{               /* downdraft
		    xevacon=0.0;
                  }
                }else{                 /* un-saturation
		  if(w[k][j]>0.0){     /* updraft
                    xevacon=0.0;
                  }else{               /* downdraft
		    xevacon=fqp[k][j];
                  }
                }
                ------------------------------------*/

                if(xvt0>SNOW_THRES){  /* snow region */
                   lh[k][j]=((clv+clf)/cp)*xevacon*3600.0;   /* [K/hr] */
                }else{
                   lh[k][j]=(clv/cp)*xevacon*3600.0;
                }
                /* lh[k][j]=(clv/cp)*xevacon*3600.0; /* not consider ice phase */

#ifdef DEBUG2
	      if(it<3){
                printf("k=%2d w=%6.1f vt=%6.1f qp=%10.2e dqp/dz=%10.2e ",
		       k,w[k][j],vt[k][j],qp[k][j],(qp[k+1][j]-qp[k-1][j])/(dzm*2.0));
                printf("advqph=%10.2e advqp=%10.2e srcqp=%10.2e | fqp=%10.2e LH=%8.2f\n",
                        advqph,advqp,srcqp,fqp[k][j],lh[k][j]);
              }
#endif

             }
          }
          /* fill the lowest (echo bottom) fqp and lh */
          lowest_flag=0;
          for(k=0; k<ZDIM-2; k++){
             if(lowest_flag==0 && qp[k][j]>UNDEF && 
                fqp[k+1][j]>UNDEF && fqp[k+2][j]>UNDEF &&
                lh[k+1][j]>UNDEF && lh[k+2][j]>UNDEF){ 
	           fqp[k][j] = 2.0*fqp[k+1][j] - fqp[k+2][j];
	           lh[k][j] = 2.0*lh[k+1][j] - lh[k+2][j];
                   lowest_flag=1;
             }
          }

   /* ---- comparison between integ-LH and srfR-LH ------------------
         Integ[rho * LH/3600]dz = (Lv/Cp) * Rsfc/3600  : Rsfc [mm/h]
                                                       : LH [K/hr]
      --------------------------------------------------------------- */
          /* vertical integration of LH */
          LHinteg=0.0;  LHintegN=0.0;
          for(k=0; k<ZDIM; k++){
	     if(lh[k][j]>UNDEF){
                 LHinteg+=rho[k]*(lh[k][j]/3600.)*dzm;
                 LHintegN +=1.0;
             }
	     if(lh[k][j]>UNDEF){
	        if(lh[k][j]<=(UNDEF+0.1)){
		  printf("########## Abnormal UNDEF value: lh=%f\n",
			 lh[k][j]);
                }
             }
          }
	  /* compare LHinteg and sfcR */
          if(sfcr[j]>0.5){
             LHsfcR = (clv/cp)*sfcr[j]/3600.;
          }else{
             LHsfcR = 0.0;
          }
          if(LHintegN > 2.0){
             difLH=LHinteg-LHsfcR;
          }else{
	     difLH=0.0;
          }

   /* ---- adjustment of the w-profile coefficients (bisection method) ---- */
          /* check the first guess */
          if(it==0 && difLH > LH_THRES){  /* usually difLH <= 0 */    
             printf("##### 1st guess ERROR at j=%2d  wtype=%1d  sfcR=%5.1f  difLH=%f",
                     j,wtype,sfcr[j],difLH);
             printf(" ----- stop the iteration\n");
	     printf("                         j=%2d  trop=%5.2f cldt=%5.2f melt=%5.2f sfcHT=%5.2f\n",
                            j,zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
             it=99;
             difLH=0.0;   /* means stop the iteration */
          }
          if(it==0)  difLH0 = difLH;
          if(it==1)  difLH1 = difLH;
          if(it > 1){
             if((difLH0*difLH1)<0.0){
	        /* bisection method */
	        it2++; 
                if((difLH*difLH0)>0.0){  /* replace the edge to midium */
	           difLH0 = difLH;
	           adjw0  = adjw;
	        }else{
                   difLH1 = difLH;
                   adjw1  = adjw;
                }
	        adjw = adjw0 + (adjw1-adjw0)*0.5;    /* set adjw on the bisection */
             }else{
                /* failed to get the effective 2nd guess (now still it2=0) */
	        if(it2>0){     /* check the bisection (maybe no need) */
		   printf("##### FAILED bisection nethod: it=%d(%d) ",it,it2);
                   printf(" difLH=%f difLH0=%f difLH1=%f \n",difLH,difLH0,difLH1);
                   difLH=0.0;   /* means stop the iteration */
                } 

               /* ====  Doushiyou..... now still it2=0
                        (1) if wtype=1  sfcR=300 mm/hr
                            else        srcR=100 mm/hr 
			(3) adjust advqph (0.0 to 1.0, 0.0 to -1.0) ===== */

	        difLH1 = difLH;

                if(it>=2 && it<9){
                   if(wtype==1){             /* conv */
                      if(sfcr[j]<0.5){
                         guessR=0.5*5.0*it;
                      }else{
                         guessR=sfcr[j]*5.0*it;
                      }
	              if(guessR>300.0) guessR = 300.0;
                   }else{                    /* strat, anvil, shallow */  
                      if(sfcr[j]<0.5){
                         guessR=0.5*2.0*it;
                      }else{
                         guessR=sfcr[j]*2.0*it;
                      }
	              if(guessR>100.0) guessR = 100.0;
                   }
                   guessLH = (clv/cp)*guessR/3600.; 
                   adjw=guessLH/LHintegS;
                }

                if(it>=9 && it<ITMAX-2){
                   if(wtype==1){             /* conv */
                      if(sfcr[j]<0.5){
                         guessR=0.5*5.0;
                      }else{
                         guessR=sfcr[j]*5.0;
                      }
	              if(guessR>300.0) guessR = 300.0;
                   }else{                    /* strat, anvil, shallow */  
                      if(sfcr[j]<0.5){
                         guessR=0.5*2.0;
                      }else{
                         guessR=sfcr[j]*2.0;
                      }
	              if(guessR>100.0) guessR = 100.0;
                   }
                   guessLH = (clv/cp)*guessR/3600.; 
                   adjw=guessLH/LHintegS;
                   if(it>=9)  adjadv=(it-8)*0.1;
                   if(it>=19) adjadv=(18-it)*0.1;
                }
	    }  /* end of else if(difLH0*difLH1<0) */

            /* case of failure in the iteration */
            if(it==ITMAX-2){
               printf("=== ITERATION FAILED at j=%2d wtype=%1d sfcR=%6.2f: ",
                       j,wtype,sfcr[j]);
               printf("Itar=%2d(%2d) wmax=%6.2f adjw=%7.4f adjadv=%4.1f ",
		       it,it2,wmax,adjw,adjadv);
               printf("w=0 at %5.2f, %5.2f,%5.2f,%5.2f\n",
                       zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
               printf("                        difLH=%6.2f difLH0=%6.2f difLH1=%6.2f  ",
		       difLH,difLH0,difLH1);
               printf("LHsfcR=%7.2f vs LHinteg=%7.2f (LHintegN=%2.0f)\n",
	               LHsfcR,LHinteg,LHintegN);

               if(it2==0){    /* if the 2nd guess has not gotten. 1st guess is used */
                  guessR  = sfcr[j];
                  if(sfcr[j]<0.5) guessR = 0.5;
                  guessLH = (clv/cp)*guessR/3600.; 
                  adjw=guessLH/LHintegS;
                  adjadv=0.0;
               }
            }            
            if(it==ITMAX-1){
	       if(it2==0){
		 printf("                        ");
	          printf("---> the 1st guess was used: wmax=%6.2f difLH=%6.2f\n",
                         wmax,difLH);
               }
            }

          }  /* end of if(it>1) */

#ifdef DEBUG1
       printf("= Itar=%2d(%2d) j=%2d wtype=%1d:  difLH=%8.3f difLH0=%8.3f difLH1=%8.3f ",
	       it,it2,j,wtype,difLH,difLH0,difLH1);
       printf("adjw=%f adjw0=%f adjw1=%f wmax=%f adjadv=%f\n",
	       adjw,adjw0,adjw1,wmax,adjadv);
       printf("       LHsfcR=%7.3f vs LHinteg=%8.3f (LHintegN=%3.0f)  ",
	       LHsfcR,LHinteg,LHintegN);
       printf("LHinteg0=%8.3f LHinteg1=%8.3f\n",
	       LHsfcR+difLH0,LHsfcR+difLH1);
#endif

          /* store the previous itaration values */
          itarN[j]=it;

          }    /* if(fabs(difLH) > LH_THRES) */

	}  /* end of iteration loop for(it=0; it<ITMAX, it++) */
/*=================================================================*/

      }    /* end of if(wtype>0) */
      if(itarN_max < itarN[j]) itarN_max = itarN[j];

#ifdef DEBUG1
      if(wtype>0){
         printf("=== CONVERGENCED at Itar Num=%2d(%2d) at j=%2d ",itarN[j],it2,j);
         printf("  wtype=%d  sfcR=%6.2f",wtype,sfcr[j]);
         printf("  difLH=%8.3f difLH0=%8.3f difLH1=%8.3f integN=%3.0f\n",
                   difLH,difLH0,difLH1,LHintegN);
      }   
#endif

      }  /* end of angle_bin loop */

       return itarN_max;
}
