/**********************************************************
                                 Dec. 17, 2001  Shin Satoh
FUNCTION: int w4pf_prof()
SOURCE FILE: w4pf_prof.c
PURPOSE: Calculate coefficients of 4th power function
         for w-profile estimation

CALLING SEQUENCE:
    (I) float *zt          : tropopause height [km] at w=0
    (I) float *zc          : cloud top height [km] at w=0
    (I) float *zm          : melting level or cloud bottom height [km]
    (O) float coef[5]      : coefficients of the 4th power function

RETURN: normal 0
************************************************************/
#include <math.h>
#include <stdio.h>
#include "glhret_v3.3.h"    /* defines function prototypes */

int w4pf_prof(double zt, double zc, double zm, float *coef)
{
      int     irow, status=0;
      float   **matrix_a, **matrix_b;

#ifdef DEBUG4
      printf("==== zt=%f   zc=%f   zm=%f\n",zt,zc,zm);
#endif

/* --- alloc matrix of fourth_power_func --- */
      matrix_a=matrix(1,3,1,3);
      matrix_b=matrix(1,3,1,1);

/* --- input matrix of simultaneous eqations --- */   
      matrix_a[1][1]=pow(zt,3.);
      matrix_a[1][2]=pow(zt,2.);
      matrix_a[1][3]=zt;
      matrix_a[2][1]=pow(zc,3.);
      matrix_a[2][2]=pow(zc,2.);
      matrix_a[2][3]=zc;
      matrix_a[3][1]=pow(zm,3.);
      matrix_a[3][2]=pow(zm,2.);
      matrix_a[3][3]=zm;

      matrix_b[1][1]=pow(zt,4.)*-1.0;
      matrix_b[2][1]=pow(zc,4.)*-1.0;
      matrix_b[3][1]=pow(zm,4.)*-1.0;

#ifdef DEBUG4
      printf("==== Result of Gauss-Jordan ====\n");
      for(irow=1; irow<4; irow++){
        printf("%10.3f %10.3f %10.3f | %10.3f\n",
	       matrix_a[irow][1],matrix_a[irow][2],matrix_a[irow][3],matrix_b[irow][1]);
      }
#endif
/* --- solve the equations using Gauss-Jordan method --- */
      status = gaussj(matrix_a, 3, matrix_b, 1);

#ifdef DEBUG4
      printf("==== Result of Gauss-Jordan ====: status=%d\n",status);
      for(irow=1; irow<4; irow++){
        printf("%10.3f %10.3f %10.3f | %10.3f\n",
	       matrix_a[irow][1],matrix_a[irow][2],matrix_a[irow][3],matrix_b[irow][1]);
      }
#endif

      if(status==0){
         coef[1]=1.0;             /* tentative value */                    
         coef[2]=matrix_b[1][1];
         coef[3]=matrix_b[2][1];
         coef[4]=matrix_b[3][1];
      }else{
         coef[1]=0.0;
         coef[2]=0.0;
         coef[3]=0.0;
         coef[4]=0.0;
      }

      return status;
}
/*-------------------------------------------------------------------------------
    Function gauss-Jordan method  (from Numerical Recipes)
-------------------------------------------------------------------------------*/
#define NR_END 1
#define FREE_ARG char*
#define SWAP(a,b) {temp=(a);(a)=(b);(b)=temp;}

int gaussj(float **a, int n, float **b, int m)
{
	int *indxc,*indxr,*ipiv;
	int i,icol,irow,j,k,l,ll;
        int status;
	float big,dum,pivinv,temp;

        status = 0;

	indxc=ivector(1,n);
	indxr=ivector(1,n);
	ipiv=ivector(1,n);
	for (j=1;j<=n;j++) {ipiv[j]=0;}
/* loop in row (i) */
	for (i=1;i<=n;i++) {
	  /* seek a pivot element */
	   big=0.0;
	   for (j=1;j<=n;j++) {
	      if (ipiv[j] != 1) {
		 for (k=1;k<=n;k++) {
		    if (ipiv[k] == 0) {
		       if (fabs(a[j][k]) >= big) {
			  big=fabs(a[j][k]);
			  irow=j;
			  icol=k;
		       }
		    } else if (ipiv[k] > 1) {
                         nrerror("gaussj: Singular Matrix-1");
                         status++;
                    }
		 }
	      }
	   }
	   ++(ipiv[icol]);

           /* change row and col */
	   if (irow != icol) {
	      for (l=1;l<=n;l++) {SWAP(a[irow][l],a[icol][l])}
	      for (l=1;l<=m;l++) {SWAP(b[irow][l],b[icol][l])}
	   }
	   indxr[i]=irow;
	   indxc[i]=icol;
	   if (a[icol][icol] == 0.0) {
              nrerror("gaussj: Singular Matrix-2");
              status++;
           }
	   pivinv=1.0/a[icol][icol];
	   a[icol][icol]=1.0;
	   for (l=1;l<=n;l++) {a[icol][l] *= pivinv;}
	   for (l=1;l<=m;l++) {b[icol][l] *= pivinv;}
	   for (ll=1;ll<=n;ll++){
	      if (ll != icol) {
		 dum=a[ll][icol];
		 a[ll][icol]=0.0;
		 for (l=1;l<=n;l++) {a[ll][l] -= a[icol][l]*dum;}
		 for (l=1;l<=m;l++) {b[ll][l] -= b[icol][l]*dum;}
	      }
           }
	}
	/* replace into the original order */
	for (l=n;l>=1;l--) {
	   if (indxr[l] != indxc[l]) {
	      for (k=1;k<=n;k++) {
		 SWAP(a[k][indxr[l]],a[k][indxc[l]]);
              }
           }
	}
	free_ivector(ipiv,1,n);
	free_ivector(indxr,1,n);
	free_ivector(indxc,1,n);

        return status;
}

int nrerror(char error_text[])
/* Numerical Recipes standard error handler */
{
	fprintf(stderr,"Numerical Recipes run-time error...\n");
	fprintf(stderr,"%s\n",error_text);
}


int *ivector(long nl, long nh)
/* allocate an int vector with subscript range v[nl..nh] */
{
	int *v;
	v=(int *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(int)));
	if (!v) nrerror("allocation failure in ivector()");
	return v-nl+NR_END;
}

float **matrix(long nrl, long nrh, long ncl, long nch)
/* allocate a float matrix with subscript range m[nrl..nrh][ncl..nch] */
{
	long i, nrow=nrh-nrl+1,ncol=nch-ncl+1;
	float **m;

	/* allocate pointers to rows */
	m=(float **) malloc((size_t)((nrow+NR_END)*sizeof(float*)));
	if (!m) nrerror("allocation failure 1 in matrix()");
	m += NR_END;
	m -= nrl;

	/* allocate rows and set pointers to them */
	m[nrl]=(float *) malloc((size_t)((nrow*ncol+NR_END)*sizeof(float)));
	if (!m[nrl]) nrerror("allocation failure 2 in matrix()");
	m[nrl] += NR_END;
	m[nrl] -= ncl;

	for(i=nrl+1;i<=nrh;i++) m[i]=m[i-1]+ncol;

	/* return pointer to array of pointers to rows */
	return m;
}

void free_ivector(int *v, long nl, long nh)
/* free an int vector allocated with ivector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}
