/**********************************************************
                                 July 05, 2002  Shin Satoh
FUNCTION: int cld_ht()
SOURCE FILE: cld_ht.c
PURPOSE: Estimate cloud top and bottom height

CALLING SEQUENCE:
    (I) float rain[][49]   : rainfall rate [mm/hr]
    (I) float *cos_antang  : cosin antenna angle
    (I) float *flag_wtype  : w-profile type 
                             (NO_RAIN=0., CONV=1., STRAT=2., ANVIL=3., SHALLOW=4.)
    (I) float *slat        : latitude [deg]
    (I) float *month_trop  : monthly tropopause height [km]
    (O) float *trop        : tropopause height [km]
    (I/O) float *cldt      : cloud top height [km]
    (I/O) float *cldb      : cloud bottom height [km]

RETURN: 0
************************************************************/
#include <math.h>
#include <stdio.h>

#define  YDIM      49      /* number of angle bins in output */
#define  ZDIM      80      /* number of vertical bins in output */
#define  N33       33      /* number of latitude for month_trop */
#define  ZRESO      0.25   /* vertical resolution in km */
#define  ZEMIN     15.0    /* minimum reflectivity [dBZ] (actually 17 dBZ, but...) */

int cld_ht (float rain[][YDIM], float *cos_antang, float *flag_wtype, float *slat,  
            float *month_trop, float *trop, float *cldt, float *cldb)
{
        int    i, ii, j, k;
        float  xdlat, ztop, zbot;

    /* get tropopause height */
        for(j=0; j<YDIM; j++){
          ii=(int)((slat[j]+40.0)/2.5);
          if(ii<0) ii=0;
          if(ii>N33-1) ii=N33-2;
          xdlat=slat[j]-(ii*2.5-40.0);
          trop[j]=month_trop[ii]+xdlat*(month_trop[ii+1]-month_trop[ii])/2.5;
#ifdef DEBUG0
          printf("j=%2d  slat=%6.2f  trop=%6.3f  (%2d %f %f)\n",
                  j+1,slat[j],trop[j],ii,month_trop[ii],month_trop[ii+1]);
#endif
        }

    /* get echo top & bottom height [km] */
    /* --- should continue more than 3 range bins --- */
        for(j=0; j<YDIM; j++){
          ztop = cldt[j];
          zbot = cldb[j];
	  for(k=0; k<ZDIM-3; k++){
            if(rain[k][j]>0.0 &&
               rain[k+1][j]>0.0 &&
               rain[k+2][j]>0.0   ){ 
                 ztop = (k+2)*ZRESO*cos_antang[j]; 
            }
          }
          if(0.9<flag_wtype[j] && flag_wtype[j]<1.1) ztop +=0.25;  /* conv */
          if(1.9<flag_wtype[j] && flag_wtype[j]<2.1) ztop +=1.0;   /* strat */
          if(2.9<flag_wtype[j] && flag_wtype[j]<3.1) ztop +=1.0;   /* anvil */
          if(3.9<flag_wtype[j] && flag_wtype[j]<4.1) ztop +=0.25;  /* shallow */
          cldt[j]=ztop;

	  for(k=ZDIM-1; k>1; k--){
            if(rain[k][j]>0.0 &&
               rain[k-1][j]>0.0 &&
               rain[k-2][j]>0.0   ){ 
                 zbot = (k-2)*ZRESO*cos_antang[j]; 
            }
          }
          cldb[j]=zbot;
       }

       return 0;     
}
