/**************************************************************
                  July 17, 2002  Shin Satoh (satoh@crl.go.jp)

Main routine of the TRMM PR Latent Heating Algorithm

SOURCE FILE: glhret_v4.1.c  <<< version 4.1 >>>
PURPOSE:     Retreive the latent heating profiles using 
             a TRMM 2A25 product
INPUT DATA:  2A25 version 5 swath data
OUTPUT DATA: grided 4-byte binary file for GrADS display
             and GrADS ctl file
COMPILE:     make -f glhret.make
USAGE:       glhret {Input file name} {GrADS Output file name}\ 
                     [start scan num] [end scan num]
             (it is possible to omit [] terms)
NOTE:        You need "param_lhret.dat" and "monthly_trop.dat"
             files on your executing directory

****************************************************************/
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "IO.h"             /* TSDIS Tool Kit */
#include "IO_PR.h"          /* TSDIS Tool Kit */
#include "IO_INTR_PR.h"     /* TSDIS Tool Kit */
#include "glhret_v4.1.h"    /* defines function prototypes */

#define  UNDEF   -9.e+33    /* missing data in output */
#define  UNDEF2  -8.9e+33   /* missing data avoiding roundoff error */
#define  ZRESO      0.25    /* vertical resolution in km */
#define  N3         3       /* number of raintype */
#define  N5         5       /* number of nodes for LWC-Z relations */
#define  N6         6       /* number of range bin numbers */
#define  N21       21       /* number of initial values for rho & vratio */
#define  N33       33       /* number of latitude for month_trop */
#define  NANGLE    49       /* number of angle bins */
#define  NRANGE    80       /* number of vertical bins */
#define  NFNAME   100       /* number of characters for file names */
#define  YDIM      49       /* number of angle bins in output (=NANGLE) */
#define  ZDIM      80       /* number of vertical bins in output (=NRANGE) */
#define  PARAM_FILE "param_lhret.dat"    /* name of the parameter file */
#define  TROP_FILE  "monthly_trop.dat"   /* name of the tropopause height file */

#define  RAD   (3.141592653589/180.0)


int main(int argc, char *argv[])
{
/* ===========================================================================
      DEFINE VARIABLES
   =========================================================================== */
/* ------- HDF/Toolkit variables ------- */
    L2A_25_SWATHDATA  read_data;
    IO_HANDLE         granuleHandle2A25;
    DATE_STR          beginDate;
    TIME_STR          beginTime;
    DATE_STR          endDate;
    TIME_STR          endTime;
    char  granuleID[NFNAME],algorithmVersion[NFNAME];
    char  filemode;
    int   orbitNumber,attitudeMode,productVersion;
    int   dataType,status,orbitSize;

/* ------- Common variables ------- */
    int   sscan,escan;
    int   i,ii,j,k,kk,l,imonth;
    float slon_nadir,first_scan_sec=0.0;

/* ------- Air parameters ------- */    
    float press_km[N21];               /* pressure at every km */
    float tempa_km[N21];               /* temparature at every km */
    float rho_km[N21];                 /* air dinsity at every km */
    float vratio_km[N21];              /* relative terminal velocities at every km */
    float press[NRANGE];               /* pressure (US standard) */
    float tempa[NRANGE];               /* temparature (US standard) */
    float qvs[NRANGE];                 /* saturated mixing ratio of vapor */
    float rho[NRANGE];                 /* air density (US standard) */
    float vratio[NRANGE];              /* relative terminal velocities */
    float month_trop[N33];             /* monthly tropopause hights in each lat */ 
    float bbht_avg, xht, xtempC, xes;

/* ------- WC-Ze relation --------- */
    char  init_file[NFNAME];           /* names of the parameter files */
    float zwc_a_c0[N3][N5];            /* LWC-Z coefficient, initial value */
    float zwc_a_c1[N3][N5];            /* a = 10^(a_c0+a_c1*epsilon+a_c2*epsilon^2) */
    float zwc_a_c2[N3][N5];            /*   where x = log10(epsilon) */
    float zwc_b_c0[N3][N5];            /* LWC-Z power, initial value */
    float zwc_b_c1[N3][N5];            /* b = 10^(b_c0+b_c1*epsilon+b_c2*epsilon^2) */
    float zwc_b_c2[N3][N5];            /*   where x = log10(epsilon) */
    float zwc_a[NRANGE];               /* coefficient in WC-Z relation */
    float zwc_b[NRANGE];               /* power in WC-Z relation */
    int   raintype;                    /* rain type (0=STRAT, 1=CONV, 2=OTHERS) */ 
    int   flag_warm;                   /* warm rain flag */

/* ------- 2A25 original/derived variable ------- */
    int   sfcBin, bbBin, cldtBin;
    int   rbinNum[N6];                 /* range bin numbers of top[0], bottom[1], 
                                          surface[2], BB[3], PIZ_threshold[4], ZeMax[5] */
    int   rbupTop, rbupCLF, rbupSurf, rbupBB;
    int   kcldt, kcldb, lowest_flag;
    int   rbinNode[N5];                /* node range bin numbers of ZR coefficients */
    float epsilonF;                    /* attenuation correction factor */
    float weightF;                     /* weighting factor in the calc of epsilon */
    float nubfCF;                      /* non-uniform beam filling correction factor */
    double xze, xz, xrain, xwc, xqp, xvt, xvt0;
    double bbhtsum, bbhtnum, rainsum, rainnum;
    double qpsum, qpnum, qpdiff, vtsum, vtnum;
    /* original 2-dims array is defined as first z-dim (i.e. rain[YDIM(2)][ZDIM(1)]) ,
       but this program uses the opposite arangement order for GrADS output */
    static float  cze[ZDIM][YDIM];
    static float  rain[ZDIM][YDIM];
    static float  wc[ZDIM][YDIM];
    static float  qp[ZDIM][YDIM];
    static float  vt[ZDIM][YDIM];
    static float  vt0[ZDIM][YDIM];
    static float  zra[ZDIM][YDIM], zrb[ZDIM][YDIM];
    static float  zwa[ZDIM][YDIM], zwb[ZDIM][YDIM];
    static float  xxrain[ZDIM], xxqp[ZDIM], xxvt[ZDIM];
    static float  sfcr[YDIM];          /* near-surface rainfall [mm/hr] */
    static float  sfcz[YDIM];          /* near-surface reflectivity [dBZ] */
    static float  sfcht[YDIM];         /* surface height [km] */
    static float  bbht[YDIM];          /* bright band height [km] */
    static float  stmht[YDIM];         /* storm height [km] from rbinNum[0] */
    static float  clfht[YDIM];         /* clutter-free height [km] from rbinNum[1] */
    static float  flag_rain[YDIM];     /* 0.=no_rain, 1.=possible, 2.=certain */
    static float  flag_PIA[YDIM];      /* 0.<=3 db, 1.>3 db, 2.>10 db */
    static float  flag_rtype[YDIM];    /* 0.=strat, 1.=conv, 2.=others */
    static float  flag_BB[YDIM];       /* 0.=no BB, 1.=BB exist, 2.=warm_rain */
    static float  flag_bottom[YDIM];   /* 0.=normal, 1.>2 km, 2.>4 km, 3.=missing */

/* ------- latent heating algorithm derived variable ------- */ 
    int   iter_max, iter_fail;
    static float  stime[YDIM];         /* scan time --> see the code */
    static float  slat[YDIM];          /* latitude [deg] */
    static float  slon[YDIM];          /* longitude [deg] */
    static float  trop[YDIM];          /* troppause height [km] */
    static float  cldt[YDIM];          /* cloud top height [km] */
    static float  cldb[YDIM];          /* cloud bottom height [km] */
    static float  w[ZDIM][YDIM];       /* vertical wind profile [m/s] */
    static float  fqp[ZDIM][YDIM];     /* production rate of qp [kg/s] */
    static float  lh[ZDIM][YDIM];      /* latent heating profile [K/hr] */
    static float  flag_wtype[YDIM];    /* w-profile type flag
                                          0.=no_rain, 1.=conv, 2.=strat, 3.=anvil, 4.=shallow */
    static float  flag_iter[YDIM];     /* iteration calcuration flag
                                          0.=normal, 1.=failed, 2.=1st guess error,  
                                          3.=2nd guess error, 4.=other error */ 
    static float  cos_antang[YDIM];    /* cosin antenna angle */

/* ------- GrADS related variables ------- */    
    FILE  *fpg;
    char  g_name[NFNAME];
    int   m, n, mc, nc;
    int   outsize1, outsize2;
    char   c_var[400], c_memo[128];
    int    nx, ny, nz, nt, tdim, iyear, iday, imon, ihr, imin;
    double xmin, ymin, zmin, tmin, dx, dy, dz, dt, no_data;


/* ===========================================================================
      OPEN FILES, READ PARAMETERS
   =========================================================================== */
/* ------- open input file (HDF) ------- */
    if(argc < 3){
       printf("Usage: %s {Input file name} {GrADS Output file name} [start scan num] [end scan num]\n",argv[0]);
       exit(0);
    }

    sscan = 1;                              /* start scan num (default) */
    if(argc >= 4) sscan = atoi(argv[3]);
    escan = 9999;                           /* end scan num (default) */
    if(argc >= 5) escan = atoi(argv[4]);
    if(escan < sscan) escan = sscan;

    strcpy(granuleID,argv[1]);	
    if(access(granuleID, 000) == -1){
        printf("Input file name error!\n");
        exit(1);
    }

    dataType = TK_L2A_25;     /* TK_L2A_25 is defined in IO.h */
    filemode = TK_READ_ONLY;

    status = TKopen(granuleID, dataType, filemode, &granuleHandle2A25);
    if(status  != TK_SUCCESS) {
	printf("Toolkit file open error!\n");
        exit(1);
    }

/* ------- open output file (GrADS) ------- */
       strcpy(g_name,argv[2]);
       if((fpg = fopen(g_name,"w" )) == NULL ){
          printf("Output file open error!: %s \n",g_name); 
          exit(1);   
       }

/* ------- check metadata  ------ */
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_BEGIN_DATE, &beginDate);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_BEGIN_TIME, &beginTime);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_END_DATE, &endDate);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_END_TIME, &endTime);
       status = TKreadMetadataChar(&granuleHandle2A25,
                        TK_GRANULE_ID, granuleID);
       status = TKreadMetadataChar(&granuleHandle2A25,
                        TK_ALGORITHM_VERSION, algorithmVersion);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_ATTITUDE_MODE, &attitudeMode);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_ORBIT_SIZE, &orbitSize);

       printf("# granule ID    = %s\n", granuleID );
       printf("# algorithm Version = %s\n", algorithmVersion );
       printf("# attitude Mode = %d\n", attitudeMode );
       printf("# begin time    = %4d/%2d/%2d, %2d:%2d:%2d \n",
                 beginDate.tkyear, beginDate.tkmonth, beginDate.tkday,
                 beginTime.tkhour, beginTime.tkminute, beginTime.tksecond );
       printf("# end time      = %4d/%2d/%2d, %2d:%2d:%2d \n",
                 endDate.tkyear, endDate.tkmonth, endDate.tkday,
                 endTime.tkhour, endTime.tkminute, endTime.tksecond );
       printf("# num of scan   = %d\n\n",  orbitSize );

       if(orbitSize < escan) escan=orbitSize;
       printf("# output scan num: %d - %d\n\n",sscan,escan);

/* ------- read parameter file ------ */
       strcpy(init_file, PARAM_FILE);
       status = read_param(init_file, zwc_a_c0, zwc_a_c1, zwc_a_c2, 
                           zwc_b_c0, zwc_b_c1, zwc_b_c2, rho_km, 
                           tempa_km, press_km, vratio_km);
       if(status > 1){
	  if(status<99) printf("data read error: %d mismatched in %s\n", 
                        status, init_file);
          if(status == 99) printf("file open error: %s\n",init_file);
          exit(1);
       } 

/* ------- read tropopause heights file ------ */
       imonth = beginDate.tkmonth;
       strcpy(init_file, TROP_FILE);
       status = read_trop(init_file, imonth, month_trop);
       if(status > 1){
	  if(status<88) printf("data read error: %d mismatched in %s\n", 
                        status, init_file);
          if(status == 88) printf("abnormal month data: %s\n",init_file);
          if(status == 99) printf("file open error: %s\n",init_file);
          exit(1);
       } 

/* ------- get air pressure, temparature, qvs, density, and vratio ------- */
       if(ZDIM>NRANGE){
         printf("dimension error: ZDIM should be smaller NRANGE\n");
         exit(1);
       }
       status = air_param(press_km, tempa_km, rho_km, vratio_km, 
                          press, tempa, qvs, rho, vratio);

/* ------- get common cos ant angle ------ */
       for(j=0; j<YDIM; j++){
          cos_antang[j] = cos((j-24)*0.71*RAD);
       }


/* ===========================================================================
      READ ACTUAL DATA and DERIVE WC, qp, Vt
   =========================================================================== */
/* ------- skip scan ------- */
       for(i=1; i<sscan; i++){
         status = TKreadScan(&granuleHandle2A25,&read_data);
         if(status != TK_SUCCESS){
            printf("### Read Error at scan num=%d\n",i);
         }
       }

/* ------- read scan by scan ------- */
       for(i=sscan; i<=escan; i++){      /* scan loop */
         status = TKreadScan(&granuleHandle2A25,&read_data) ;
         if(status != TK_SUCCESS){
            printf("### Read Error at scan num=%d\n",i);
         }else{
	   ii=i-sscan;    /* ii means output scan number, from 0 to escan-sscan */

	 /*--- get scan time and local time etc ---*/ 
         stime[0] = (float)read_data.scanTime;
         stime[1] = (float)beginDate.tkyear;
         stime[2] = (float)beginDate.tkmonth;
         stime[3] = (float)beginDate.tkday;
         stime[4] = (float)((int)(stime[0]/3600.));
         stime[5] = (float)((int)((stime[0]-stime[4]*3600.)/60.));
         stime[6] = (float)((int)(stime[0]-stime[4]*3600.-stime[5]*60.));
         slon_nadir = (float)read_data.geolocation[24][1];
	 stime[10] = stime[0]+(12.0*3600.0*slon_nadir/180.0);   /* local time in sec*/
         stime[11] = stime[1];
         stime[12] = stime[2];
         stime[13] = stime[3];
         if(stime[10]<0.0){
            stime[13] = stime[3]-1.0;
            stime[10] += 24.0*3600.0; 
         }
         if(stime[10]>24.0*3600.0){
            stime[13] = stime[3]+1.0;
            stime[10] -= 24.0*3600.; 
         }
         stime[14] = (float)((int)(stime[10]/3600.));  /* local time in hour */
         stime[15] = (float)((int)((stime[10]-stime[14]*3600.)/60.));
         stime[16] = (float)((int)(stime[10]-stime[14]*3600.-stime[15]*60.));
         for(j=7; j<10; j++) stime[j]=UNDEF;
         for(j=17; j<49; j++) stime[j]=UNDEF;


/* ------- angle bin loop ------- */
         for(j=0; j<NANGLE; j++){
            slat[j]  = (float)read_data.geolocation[j][0];
            slon[j]  = (float)read_data.geolocation[j][1];

            /*--- get rainfall flag in 2A25 ------------------------------
	          0: (bit  1) no rain
	   	 +1: (bit  1) rain possible
	   	 +2: (bit  2) rain certain
	   	 +4: (bit  3) zeta^beta > 0.5 (PIA larger than 3 dB)
	   	 +8: (bit  4) large attenuation (PIA larger than 10 dB)
	        +16: (bit  5) stratiform
	        +32: (bit  6) convective
	        +64: (bit  7) BB exist
	       +128: (bit  8) warm rain
	       +256: (bit  9) rain bottom above 2 km
	       +512: (bit 10) rain bottom above 4 km
	     +16384: (bit 15) data missing between rain top and bottom
	    -------------------------------------------------------------*/
            /*--- output flag as 2A25 original --------------------------
               flag_rain[49]:    0.=no_rain, 1.=possible, 2.=certain
               flag_PIA[49]:     0.<=3 db, 1.>3 db, 2.>10 db
	       flag_rtype[49]:   0.=strat, 1.=conv, 2.=others <<original>>
               flag_BB[49]:      0.=no_rain?, 1.=BB exist, 2.=warm_rain
               flag_rbottom[49]: 0.=normal, 1.>2 km, 2.>4 km, 3.=missing
 	    -------------------------------------------------------------*/
            flag_rain[j]=0.0;  flag_PIA[j]=0.0;  flag_rtype[j]=2.0;  
            flag_BB[j]=0.0;   flag_bottom[j]=0.0;
            if( (read_data.rainFlag[j]&1)      ==1)    flag_rain[j]=1.0;
            if(((read_data.rainFlag[j]&2)>>1)  ==1)    flag_rain[j]=2.0;
            if(((read_data.rainFlag[j]&4)>>2)  ==1)    flag_PIA[j]=1.0;
            if(((read_data.rainFlag[j]&8)>>3)  ==1)    flag_PIA[j]=2.0;
            if(((read_data.rainFlag[j]&16)>>4) ==1)    flag_rtype[j]=0.0;
            if(((read_data.rainFlag[j]&32)>>5) ==1)    flag_rtype[j]=1.0;
            if(((read_data.rainFlag[j]&64)>>6) ==1)    flag_BB[j]=1.0;
            if(((read_data.rainFlag[j]&128)>>7)==1)    flag_BB[j]=2.0;
            if(((read_data.rainFlag[j]&256)>>8)==1)    flag_bottom[j]=1.0;
            if(((read_data.rainFlag[j]&512)>>9)==1)    flag_bottom[j]=2.0;
            if(((read_data.rainFlag[j]&16384)>>14)==1) flag_bottom[j]=3.0;

            /*--- get surface & BB height [km] --------------------------
               range bin numbers of top[0], bottom[1],surface[2], BB[3],
                                    PIZ_threshold[4], ZeMax[5]
	    -------------------------------------------------------------*/
            for(l=0; l<N6; l++){
               rbinNum[l] = (int)read_data.rangeBinNum[j][l];
            }
            rbupTop    = 79-rbinNum[0];     /* Top range bin num upward */
            rbupCLF    = 79-(rbinNum[1]-1); /* Clutter-free range bin num upward */
            rbupSurf   = 79-rbinNum[2];     /* Surf range bin num upward */
            rbupBB     = 79-rbinNum[3];     /* BB range bin num upward */
            /* Memo: - rbinNum[0] is defined as the 1 km above the 
                       rain-certain bin (in ver. 5.53), but it may be 
                       the highest obs range bin if it is no-rain.
	             - rbinNum[1] is the top of the possible surface
                       cluttered range. Hence, "rbinNum[1]-1" is 
                       the actual valid rain bottom data. The near-
                       surface-rain must be the same as rain[rbinNum[1]-1]
            */
            stmht[j] = rbupTop*ZRESO*cos_antang[j];  /* storm top height */
            clfht[j] = rbupCLF*ZRESO*cos_antang[j];  /* clutter-free height */
            sfcht[j] = rbupSurf*ZRESO*cos_antang[j]; /* actual surface height */
            bbht[j]  = rbupBB*ZRESO*cos_antang[j];   /* bright band height */
            cldt[j] = rbupTop*ZRESO*cos_antang[j];   /* temporal(initial) cloud top height */
            cldb[j] = rbupCLF*ZRESO*cos_antang[j];   /* temporal(initial) cloud bottom height */

            if(flag_rain[j] < 1.1){    /* no-rain or rain-possible */
	       cldt[j] = 0.0;
	       cldb[j] = 0.0;
            }

            if(bbht[j]>0.0){
               bbht[j] = bbht[j]*(5./6.) - 0.5;   /* bbht is defined as +3 dec.C */
	    }else{
	       bbht[j]=0.0;
            }
/**********************************************************
==> rangeBinNum2r4.c <==    refered from 2A25 USERS GUIDE
Version 5.5 of 2A25 

NOTE:
  The freezing height correction factor (fhcf) is introduced here 
because the freezing level from 2A-23 is a rather conservative (low end) 
estimate in order to avoid a wrong identification of warm rain.  
2A-25 uses the freezing estimate to determine the k-Z and Z-R coefficients.  
For this purpose, it is probably better to over-estimate the freezing height 
than to under-estimate it.  The current version of 2A-23 uses 5 deg.C/km as 
the lapse rate for the conservative estimate.  We change it to 6 deg.C/km
in this program. Hence the correction factor is 6/5=1.2.  
    The freezing height correction facotr (fhcf) is now written in the 
parameter file.  It is read in the main() and transfered into this function 
as an argument. Log: startDist added on 16 Dec. 1997.

====> Since this is special purpose for only 2A25, 
      I should use 2A23 original freezing height.

* 2A23 ver 5.4 uses 6 deg.C/km
* 2A23 defines the freezing height (freezH) as 0 C level
************************************************************/


            /*--- new flag (flag_wtype) to choice the w-profile shape ----
                flag_wtype 
                0.=NO_RAIN, 1.=CONV, 2.=STRAT, 3.=ANVIL, 4.=SHALLOW
 	    -------------------------------------------------------------*/
            /* if(flag_rain[j] > 0.9){                      /* rain possible */
            if(flag_rain[j] > 1.9){                         /* rain certain */
               if(flag_rtype[j]<0.1){                       /* original strat */
                  if(rbupTop>=rbupBB) flag_wtype[j]=2.0;  
                  if(rbupTop<rbupBB)  flag_wtype[j]=4.0;
               }
               if(0.9<flag_rtype[j] && flag_rtype[j]<1.1){  /* original conv */
		  flag_wtype[j]=1.0;
               }
               if(1.9<flag_rtype[j] && flag_rtype[j]<2.1){  /* original others */
		  flag_wtype[j]=3.0;                        /* anvil */
                  if(rbupTop<rbupBB)  flag_wtype[j]=4.0;    /* new flag of shallow */
               }
            }else{
	       flag_wtype[j]=0.0;  /* no_rain */
            }

            /*--- Z-WC relation using a new raintype <not use in v.4.1>----
                 (-1=NO_RAIN, 0=STRAT/ANVIL, 1=CONV, 2=SHALLOW) 
                 in this routine, anvil => STRAT, shallow => OTHERS
            -------------------------------------------------------------*/
            raintype=-1;     /* including rain possible */
            if(((read_data.rainFlag[j]&16)>>4) ==1) raintype=0; /* 2A25 strat */
            if(((read_data.rainFlag[j]&32)>>5) ==1) raintype=1; /* 2A25 conv */
            if(((read_data.rainFlag[j]&128)>>7)==1) raintype=2; /* 2A23 warm rain */
            if(2.9<flag_wtype[j] && flag_wtype[j]<3.1) raintype=0;  /* anvil */
            if(3.9<flag_wtype[j] && flag_wtype[j]<4.1) raintype=2;  /* shallow */
            /*--- flag_warm (1=warm rain) for Z-WC relation ---*/
            flag_warm =0;
            if(raintype==2) flag_warm=1;

            /*--- Original Z-WC relation using 2A25 raintype --------------
                 (-1=NO_RAIN, 0=STRAT, 1=CONV, 2=OTHERS) 
                 in this routine, anvil => STRAT, shallow => OTHERS
	    -------------------------------------------------------------*/
            raintype=-1;
            if( (read_data.rainFlag[j]&1)      ==1) raintype=0;  /* rain possible */
            if(((read_data.rainFlag[j]&16)>>4) ==1) raintype=0;  /* 2A25 strat */
            if(((read_data.rainFlag[j]&32)>>5) ==1) raintype=1;  /* 2A25 conv */
            if(((read_data.rainFlag[j]&128)>>7)==1) raintype=2;  /* 2A25 others */
            flag_warm =0;
            if(((read_data.rainFlag[j]&128)>>7)==1) flag_warm=1;
            /* the original warm flag was defined by 1.5 km below the BB */


            /*--- get coefficients of Z-WC relation ---*/
            nubfCF   = read_data.nubfCorrectFactor[j][1];
	    epsilonF = read_data.epsilon[j];
	    weightF  = read_data.weightW[j];
            /*-------------------------------------------------------- 
              If you use 2A25 version 4 product, remove the commentted 
              following 2 lines to use the fixed epsilon.  
                  epsilonF = 1.0;
                  weightF = 0.0;
	      ---------------------------------------------------------*/ 

            /*--- get range bin number of the 5 Node ---*/
            for(l=0; l<N5; l++){
               rbinNode[l] = (int)read_data.ZRParmNode[j][l];
            }

            status = wc_param(rbinNum, raintype, flag_warm, zwc_a_c0, zwc_a_c1, zwc_a_c2, 
                              zwc_b_c0, zwc_b_c1, zwc_b_c2, nubfCF, epsilonF, 
                              weightF, rbinNode, zwc_a, zwc_b);

#ifdef DEBUG
            if(j==24){
               printf("# ii=%4d  nubfCF=%6.4f  epsilonF=%6.4f  weightF=%6.4f",
		      ii, nubfCF, epsilonF, weightF);
               printf("  rbinNode[0]=%2d  [1]=%2d  [2]=%2d  [3]=%2d  [4]=%2d\n",
		      rbinNode[0],rbinNode[1],rbinNode[2],rbinNode[3],rbinNode[4]);
               printf("                                                  ");
               printf("  rbinNum[0]=%2d  [1]=%2d  [2]=%2d  [3]=%2d  [4]=%2d  [5]=%2d\n",
		      rbinNum[0],rbinNum[1],rbinNum[2],rbinNum[3],rbinNum[4],rbinNum[5]);
            }
            if(j==24 && raintype!=-1){ 
               for(k=rbinNum[0]; k<rbinNum[1]; k++){
	         printf("# j=%2d  k=%2d  zwc_a=%10.4f  zwc_b=%10.4f\n",j,k,zwc_a[k],zwc_b[k]);
               }
	    }
#endif

            /*--- get surface Ze & rain ---*/
            if(flag_rain[j]>1.9){         /* rain certain */
               sfcz[j] = (float)read_data.nearSurfZ[j];
               sfcr[j] = (float)read_data.nearSurfRain[j];
            }else{
               sfcz[j] = UNDEF;
               sfcr[j] = UNDEF;
            }

            /*--- init 2D values ---*/
            for(k=0; k<NRANGE; k++){
	       cze[k][j]  = UNDEF;   /* clutter data is UNDEF */
               rain[k][j] = UNDEF;
               wc[k][j] = UNDEF;
               qp[k][j] = UNDEF;
               vt[k][j] = UNDEF;
               vt0[k][j] = UNDEF;
            }

            /*--- get Ze, Rain, WC, Vt ---------------------------------------*/
	    /* k is downward, which includes read_data.* (2A25), zwc_a, zwc_b */
	    /* kk is upward, which includes rho, vratio, and output variables */
            /* xrain and xze data includes missing value (-99.9) and          */
	    /*                             clutter value (-88.9)              */
            /*----------------------------------------------------------------*/
            for(k=0; k<NRANGE; k++){
               kk=NRANGE-1-k;                          /* the same as 79-k */
               kk = (int)(kk*cos_antang[j]+0.5);       /* Debugged on July 14 */
               xrain = (float)read_data.rain[j][k];             /* [mm/hr] */
               xze  = (float)read_data.correctZFactor[j][k];    /* [dBZ] */
               if(xze < 0.0) xze = 0.0;
               xz = pow(10.0,xze/10.0);                         /* [mm^6/m^3] */
               xwc = zwc_a[k] * pow(xz,zwc_b[k]);               /* [g/m^3] */
               xqp = xwc/rho[kk];                               /* [g/kg] */
               if(xwc > 0.0 && xrain > 0.0){
                  xvt = xrain/(3.6*xwc);                        /* [m/s] */
               }else{
		  xvt = 0.0;
               }
               xvt0 = xvt/vratio[kk];                           /* [m/s] */
               if(xrain >= 0.0 && flag_rain[j]>1.9){
                  cze[kk][j]  = xze;
                  rain[kk][j] = xrain;
                  wc[kk][j] = xwc * 1.0e-3;       /* into [kg/m^3] */
                  qp[kk][j] = xqp * 1.0e-3;       /* into [kg/kg] */
                  vt[kk][j] = xvt * (-1.0);       /* upward is plus */
                  vt0[kk][j] = xvt0 * (-1.0);     /* upward is plus */
               }
	    }

            /*--- print scan num, date, and time ---*/
            if(first_scan_sec==0.0) first_scan_sec = stime[0];
            if(j==24 && i==sscan){       /* on nadir */
               printf("First scan: %4d (t=%4d)  %4d/%2d/%2d,%2d:%2d:%2d ",
		       i,i-sscan,(int)stime[1],(int)stime[2],(int)stime[3],
		                 (int)stime[4],(int)stime[5],(int)stime[6]);
	       printf("(%2d,%2d:%2d:%2d LST) lat=%7.3f lon=%8.3f (at nadir)\n",
                       (int)stime[13],(int)stime[14],(int)stime[15],(int)stime[16],
                       slat[j],slon[j]);
            }
            if(j==24 && i==escan){       /* on nadir */
               printf("Last scan : %4d (t=%4d)  %4d/%2d/%2d,%2d:%2d:%2d ",
		       i,i-sscan,(int)stime[1],(int)stime[2],(int)stime[3],
		                 (int)stime[4],(int)stime[5],(int)stime[6]);
	       printf("(%2d,%2d:%2d:%2d LST) lat=%7.3f lon=%8.3f (at nadir)\n",
                       (int)stime[13],(int)stime[14],(int)stime[15],(int)stime[16],
                       slat[j],slon[j]);
            }

         }  /* end of angle loop */
         }  /* if-else of readScan error */

/* ===========================================================================
      GET CLOUD TOP and BOTTOM HEIGHTS
   =========================================================================== */
         /*--- get cloud top and bottom height ---*/
         status = cld_ht(rain, cos_antang, flag_wtype, slat,
                         month_trop, trop, cldt, cldb);

         for(j=0; j<YDIM; j++){
            /* eliminate abnormal surface height [km] */ 
            if(sfcht[j]>=cldt[j] || sfcht[j]>8.848){
               sfcht[j]=0.0;
               cldt[j]=0.0;
               cldb[j]=0.0;
               flag_wtype[j]=0.0;
            }

            /* eliminate abnormal cldt, cldb height */
            if(cldb[j]<sfcht[j]){
               cldb[j]=sfcht[j];
            }

            if(cldb[j]>=trop[j]){
#ifdef DEBUG
	       printf("#==[cldb>trop]== j=%2d wtype=%2.1f trop=%5.2f cldt=%5.2f ",
                       j,flag_wtype[j],trop[j],cldt[j]);
	       printf("bbht=%5.2f cldb=%5.2f sfcht=%5.2f\n",bbht[j],cldb[j],sfcht[j]);
#endif
	       cldt[j]=0.0;
	       cldb[j]=0.0;
               flag_wtype[j]=0.0;
            }

            if(cldt[j]>=trop[j]){
#ifdef DEBUG
	       printf("#==[cldt>trop]== j=%2d wtype=%2.1f trop=%5.2f cldt=%5.2f ",
                       j,flag_wtype[j],trop[j],cldt[j]);
	       printf("bbht=%5.2f cldb=%5.2f sfcht=%5.2f\n",
                       bbht[j],cldb[j],sfcht[j]);
#endif
               if(0.9<flag_wtype[j] && flag_wtype[j]<1.1){  /* conv */
                 trop[j]=cldt[j]+ZRESO;
               }
               if(1.9<flag_wtype[j] && flag_wtype[j]<3.1){  /* strat or anvil */
                  if((cldt[j]-trop[j])<ZRESO){
                     trop[j]=cldt[j]+ZRESO;
                  }else{
	             cldt[j]=0.0;
	             cldb[j]=0.0;
                     flag_wtype[j]=0.0;
                  }
               }
               if(3.9<flag_wtype[j] && flag_wtype[j]<4.1){  /* shallow */
	          cldt[j]=0.0;
	          cldb[j]=0.0;
                  flag_wtype[j]=0.0;
               }

            }

         }

/* ===========================================================================
      ELIMINATE UNNECESSARY qp and AVERAGE qp and vt
   =========================================================================== */
         for(j=0; j<YDIM; j++){
	   kcldt = (int)(cldt[j]/(ZRESO*cos_antang[j]));    /* 0 <= kcldt <= 79 */
	   kcldb = (int)(cldb[j]/(ZRESO*cos_antang[j]));
           /*--- eliminate qp above the cldt, under the cldb---*/
           for(k=0; k<kcldb+1; k++){
             qp[k][j]=UNDEF;
           }
           for(k=kcldt; k<ZDIM; k++){
             qp[k][j]=UNDEF;
           }

           /*--- eliminate BB of qp using the 5 bins average of qp ---*/
           for(k=0; k<ZDIM; k++){
             xxqp[k]=qp[k][j];
           }
           for(k=kcldb+2; k<kcldt-2; k++){
             qpsum=0.0;  qpnum=0.0;
             for(kk=k-2; kk<k+3; kk++){
               if(xxqp[kk]>=0.0){
                 qpsum += xxqp[kk];
                 qpnum += 1.0;
               }
             }
             if(qpnum>1.0){
               qpdiff = (xxqp[k]-(qpsum/qpnum))*1000.0;  /* [g/kg] */
               if(qpdiff>0.2){
                 qp[k][j]=qpsum/qpnum;
#ifdef DEBUG3
                 printf("## j=%2d  k=%2d  xxqp=%f  avg=%f  diff=%f\n",
			j, k, xxqp[k], qpsum/qpnum, qpdiff);
#endif
               }
             }
           }

           /*--- 5 bins averaging of rain, qp and vt ---*/
           for(k=0; k<ZDIM; k++){
             xxrain[k]=rain[k][j];
             xxqp[k]=qp[k][j];
             xxvt[k]=vt[k][j];
           }
           for(k=kcldb+2; k<kcldt-2; k++){
             rainsum=0.0;  rainnum=0.0;
             qpsum=0.0;    qpnum=0.0;
             vtsum=0.0;    vtnum=0.0;
             for(kk=k-2; kk<k+3; kk++){
               if(xxrain[kk]>=0.0){
                 rainsum += xxrain[kk];
                 rainnum += 1.0;
               }
               if(xxqp[kk]>=0.0){
                 qpsum += xxqp[kk];
                 qpnum += 1.0;
               }
               if(xxvt[kk]>UNDEF){
                 vtsum += xxvt[kk];
                 vtnum += 1.0;
               }
             }
             /* if(rainnum>1.0) rain[k][j]=rainsum/rainnum;  --- KEEP rain as org value */
             if(qpnum>1.0) qp[k][j]=qpsum/qpnum;
             if(rainnum>1.0) vt[k][j]=-1.0*(rainsum/rainnum)/(3600.0*qp[k][j]*rho[k]);
             /* if(vtnum>1.0) vt[k][j]=vtsum/vtnum; */

#ifdef DEBUG3
             printf("## j=%2d  k=%2d  qp=%f %f  qpsum=%f %f  vt=%f %f  vtsum=%f %f\n",
                     j, k, qp[kk][j], xxqp[kk], qpsum, qpnum, 
		    vt[kk][j], xxvt[kk], vtsum, vtnum);
#endif
           }

           /*--- fill lowest/2nd-lowest qp and vt ---*/
           lowest_flag=0;
           for(k=0; k<ZDIM-3; k++){
              if(lowest_flag==0 &&
                 qp[k][j]>=0.0 && qp[k+1][j]>=0.0 &&
                 qp[k+2][j]>=0.0 && qp[k+3][j]>=0.0){
	            qp[k+1][j] = qp[k+3][j] + 2.0*(qp[k+2][j]-qp[k+3][j]);
	            qp[k][j] =   qp[k+3][j] + 3.0*(qp[k+2][j]-qp[k+3][j]);
	            vt[k+1][j] = vt[k+3][j] + 2.0*(vt[k+2][j]-vt[k+3][j]);
	            vt[k][j] =   vt[k+3][j] + 3.0*(vt[k+2][j]-vt[k+3][j]);
                    lowest_flag=1;
              }
           } 

           /*--- re-make vt0 ---*/
           for(k=0; k<ZDIM; k++){
              kk=NRANGE-1-k;
              kk = (int)(kk*cos_antang[j]+0.5);
              if(vt[k][j]>UNDEF2){ 
                 vt0[k][j]=vt[k][j]/vratio[kk];
              }else{
		 vt0[k][j]=UNDEF;
              }
           }

         }  /* end of j-loop (j<YDIM) */

/* ===========================================================================
      RETRIEVE w-, Fqp-, LH-profile
   =========================================================================== */
         /*--- retrieve LH profile ---*/
         iter_max = lh_prof(rho, qvs, qp, vt, vt0, sfcr, 
		  	        sfcht, bbht, trop, cldt, cldb,  
                                flag_wtype, w, fqp, lh, flag_iter);

         iter_fail=0;  /* number of failures in iteration (0-49) */
         for(j=0; j<YDIM; j++){
           if(flag_iter[j]>0.0) iter_fail +=1;
         }
         if(iter_fail>0){
           printf("------ %2d times failures occured (max iter=%2d) in the scan: ",
                   iter_fail,iter_max);
           printf("%4d (t=%4d)  %2d:%2d:%2d  lat=%7.3f lon=%8.3f ------\n",
                   i,i-sscan,(int)stime[4],(int)stime[5],(int)stime[6],slat[24],slon[24]);
         }


/* ===========================================================================
      OUTPUT DATA
   =========================================================================== */
          /*--- output GrADS file ---*/
          outsize1 = YDIM * sizeof(float);
          outsize2 = YDIM * ZDIM * sizeof(float);
          fwrite(cze,1,outsize2,fpg);
          fwrite(rain,1,outsize2,fpg);
          fwrite(wc,1,outsize2,fpg);
          fwrite(qp,1,outsize2,fpg);
          fwrite(vt,1,outsize2,fpg);
          fwrite(vt0,1,outsize2,fpg);
          fwrite(w,1,outsize2,fpg);
          fwrite(fqp,1,outsize2,fpg);
          fwrite(lh,1,outsize2,fpg);
          fwrite(stime,1,outsize1,fpg);
          fwrite(slat,1,outsize1,fpg);
          fwrite(slon,1,outsize1,fpg);
          fwrite(sfcz,1,outsize1,fpg);
          fwrite(sfcr,1,outsize1,fpg);
          fwrite(stmht,1,outsize1,fpg);
          fwrite(clfht,1,outsize1,fpg);
          fwrite(sfcht,1,outsize1,fpg);
          fwrite(bbht,1,outsize1,fpg);
          fwrite(trop,1,outsize1,fpg);
          fwrite(cldt,1,outsize1,fpg);
          fwrite(cldb,1,outsize1,fpg);
          fwrite(flag_rain,1,outsize1,fpg);
          fwrite(flag_PIA,1,outsize1,fpg);
          fwrite(flag_rtype,1,outsize1,fpg);
          fwrite(flag_BB,1,outsize1,fpg);
          fwrite(flag_bottom,1,outsize1,fpg);
          fwrite(flag_wtype,1,outsize1,fpg);
          fwrite(flag_iter,1,outsize1,fpg);

       }    /* end of scan loop */


       /*--- close HDF file ---*/
       status = TKclose(&granuleHandle2A25);

       /*--- close the output file ---*/
       fclose(fpg);
       tdim = escan-sscan;
       printf("\n");
       printf("output data: %d(angle) * %d(range) * %d(scan) * 4 bytes\n",
               YDIM,ZDIM,tdim);
       printf("             3D(9) :ze,rain,wc,qp,vt,vt0,w,fqp,lh\n");
       printf("             2D(19):stime,slat,slon,sfcz,sfcr,stmht,clfht,sfcht,bbht,trop,cldt,cldb,flag(7)\n\n");

        /*--- output GrADS ctl file ---*/
        strcpy(c_memo,"2A25 Latent Heating Profile");
        strcpy(c_var,"ze,rain,wc,qp,vt,vt0,w,fqp,lh,stime,slat,slon,sfcz,sfcr,stmht,clfht,sfcht,bbht,trop,cldt,cldb,f_rain,f_PIA,f_rtype,f_BB,f_bottom,f_wtype,f_iter");
        nx=1;          ny=YDIM;     nz=ZDIM;    nt=tdim;
        xmin=0.0;      ymin=0.0;    zmin=0.0;   tmin=0.0;
        dx=4.2;        dy=4.2;      dz=ZRESO;   dt=1.0;
        no_data= UNDEF;
        iyear  = beginDate.tkyear; 
        imon   = beginDate.tkmonth;
        iday   = beginDate.tkday;
        ihr    =(int)(first_scan_sec/3600);
        imin   =(int)((first_scan_sec-ihr*3600)/60);

        ggrads_ctl( g_name, nx, ny, nz, nt, xmin, ymin, zmin, tmin,
                    dx, dy, dz, dt, no_data, ihr, imin, iday, imon, iyear,
                    c_var, c_memo );

	return(0);
}
