/**************************************************************
                   Aug 06, 2002  Shin Satoh (satoh@crl.go.jp)

Main routine of the TRMM PR Latent Heating Algorithm in level 3

SOURCE FILE: glhret_grid.c
             << based on glhret_v4.1.c >>
PURPOSE:     To calculate the gridded 3-dim structure of 
             the latent heating from multiple 2A25 granul data

INPUT DATA:  2A25 version 5 swath data
OUTPUT DATA: 4-byte binary and ctl files for GrADS

COMPILE:     make -f glhret_grid.make
USAGE:       glhret_grid {input files table(*)}
RUN:         You need "param_lhret.dat" and "monthly_trop.dat"
             files on your executing directory
             
* an example of the "input files table" as a text file
* > ls -1 ../GAME_sub/*1998050* > f_list9805a
../GAME_sub/PR2A25.19980501.02434
../GAME_sub/PR2A25.19980501.02435
../GAME_sub/PR2A25.19980501.02436
        :
        :
../GAME_sub/PR2A25.19980510.02588

****************************************************************/
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "IO.h"             /* TSDIS Tool Kit */
#include "IO_PR.h"          /* TSDIS Tool Kit */
#include "IO_INTR_PR.h"     /* TSDIS Tool Kit */
#include "glhret_grid.h"    /* defines function prototypes */

/*--- Global ---
#define  LON1    -180.0     /* output west edge in longitude [deg] *
#define  LON2     180.0     /* output east edge in longitude [deg] *
#define  LAT1     -40.0     /* output south edge in latitude [deg] *
#define  LAT2      40.0     /* output north edge in latitude [deg] *
#define  XGRID      0.5     /* output grid resolution in longitude [deg] *
#define  YGRID      0.5     /* output grid resolution in latitude [deg] *
#define  ZGRID      0.25    /* output grid resolution in height [km] *
#define  NX       720       /* NX = (int)(LON2-LON1)/XGRID *
#define  NY       160       /* NY = (LAT2-LAT1)/YGRID *
#define  NZ        80       /* NZ = 20.0/ZGRID */

/*--- Kodama's project ---*/
#define  LON1      60.0
#define  LON2     180.0
#define  LAT1     -40.0
#define  LAT2      40.0
#define  XGRID      0.1
#define  YGRID      0.1
#define  ZGRID      0.5
#define  NX      1200
#define  NY       800
#define  NZ        40

/*--- GAME-Subset ---
#define  LON1      80.0
#define  LON2     160.0
#define  LAT1      -5.0
#define  LAT2      40.0
#define  XGRID      0.5
#define  YGRID      0.5
#define  ZGRID      0.25
#define  NX       160
#define  NY        90
#define  NZ        80
--------------------------*/


#define  UNDEF   -9.e+33    /* missing data in output */
#define  UNDEF2  -8.9e+33   /* missing data avoiding roundoff error */
#define  ZRESO      0.25    /* vertical resolution in km */
#define  N3         3       /* number of raintype */
#define  N5         5       /* number of nodes for LWC-Z relations */
#define  N6         6       /* number of range bin numbers */
#define  N21       21       /* number of initial values for rho & vratio */
#define  N33       33       /* number of latitude for month_trop */
#define  NANGLE    49       /* number of angle bins */
#define  NRANGE    80       /* number of vertical bins */
#define  NFNAME   100       /* number of characters for file names */
#define  YDIM      49       /* number of angle bins in output (=NANGLE) */
#define  ZDIM      80       /* number of vertical bins in output (=NRANGE) */
#define  PARAM_FILE "param_lhret.dat"    /* name of the parameter file */
#define  TROP_FILE  "monthly_trop.dat"   /* name of the tropopause height file */

#define  RAD   (3.141592653589/180.0)


int main(int argc, char *argv[])
{
/* ===========================================================================
      DEFINE VARIABLES
   =========================================================================== */
/* ------- HDF/Toolkit variables ------- */
    L2A_25_SWATHDATA  read_data;
    IO_HANDLE         granuleHandle2A25;
    DATE_STR          beginDate;
    TIME_STR          beginTime;
    DATE_STR          endDate;
    TIME_STR          endTime;
    char  granuleID[NFNAME],algorithmVersion[NFNAME];
    char  filemode;
    int   orbitNumber,attitudeMode,productVersion;
    int   dataType,status,orbitSize;

/* ------- Common variables ------- */
    FILE  *fp;
    char  f_name[NFNAME], fl_name[NFNAME];
    int   sscan,escan;
    int   lp,i,ii,j,k,kk,l,imonth;
    float slon_nadir,first_scan_sec=0.0;

/* ------- Air parameters ------- */    
    float press_km[N21];               /* pressure at every km */
    float tempa_km[N21];               /* temparature at every km */
    float rho_km[N21];                 /* air dinsity at every km */
    float vratio_km[N21];              /* relative terminal velocities at every km */
    float press[NRANGE];               /* pressure (US standard) */
    float tempa[NRANGE];               /* temparature (US standard) */
    float qvs[NRANGE];                 /* saturated mixing ratio of vapor */
    float rho[NRANGE];                 /* air density (US standard) */
    float vratio[NRANGE];              /* relative terminal velocities */
    float month_trop[N33];             /* monthly tropopause hights in each lat */ 
    float bbht_avg, xht, xtempC, xes;

/* ------- WC-Ze relation --------- */
    char  init_file[NFNAME];           /* names of the parameter files */
    float zwc_a_c0[N3][N5];            /* LWC-Z coefficient, initial value */
    float zwc_a_c1[N3][N5];            /* a = 10^(a_c0+a_c1*epsilon+a_c2*epsilon^2) */
    float zwc_a_c2[N3][N5];            /*   where x = log10(epsilon) */
    float zwc_b_c0[N3][N5];            /* LWC-Z power, initial value */
    float zwc_b_c1[N3][N5];            /* b = 10^(b_c0+b_c1*epsilon+b_c2*epsilon^2) */
    float zwc_b_c2[N3][N5];            /*   where x = log10(epsilon) */
    float zwc_a[NRANGE];               /* coefficient in WC-Z relation */
    float zwc_b[NRANGE];               /* power in WC-Z relation */
    int   raintype;                    /* rain type (0=STRAT, 1=CONV, 2=OTHERS) */ 
    int   flag_warm;                   /* warm rain flag */

/* ------- 2A25 original/derived variable ------- */
    int   sfcBin, bbBin, cldtBin;
    int   rbinNum[N6];                 /* range bin numbers of top[0], bottom[1], 
                                          surface[2], BB[3], PIZ_threshold[4], ZeMax[5] */
    int   rbupTop, rbupCLF, rbupSurf, rbupBB;
    int   kcldt, kcldb, lowest_flag;
    int   rbinNode[N5];                /* node range bin numbers of ZR coefficients */
    float epsilonF;                    /* attenuation correction factor */
    float weightF;                     /* weighting factor in the calc of epsilon */
    float nubfCF;                      /* non-uniform beam filling correction factor */
    double xze, xz, xrain, xwc, xqp, xvt, xvt0;
    double bbhtsum, bbhtnum, rainsum, rainnum;
    double qpsum, qpnum, qpdiff, vtsum, vtnum;
    /* original 2-dims array is defined as first z-dim (i.e. rain[YDIM(2)][ZDIM(1)]) ,
       but this program uses the opposite arangement order for GrADS output */
    static float  cze[ZDIM][YDIM];
    static float  rain[ZDIM][YDIM];
    static float  wc[ZDIM][YDIM];
    static float  qp[ZDIM][YDIM];
    static float  vt[ZDIM][YDIM];
    static float  vt0[ZDIM][YDIM];
    static float  zra[ZDIM][YDIM], zrb[ZDIM][YDIM];
    static float  zwa[ZDIM][YDIM], zwb[ZDIM][YDIM];
    static float  xxrain[ZDIM], xxqp[ZDIM], xxvt[ZDIM];
    static float  sfcr[YDIM];          /* near-surface rainfall [mm/hr] */
    static float  sfcz[YDIM];          /* near-surface reflectivity [dBZ] */
    static float  sfcht[YDIM];         /* surface height [km] */
    static float  bbht[YDIM];          /* bright band height [km] */
    static float  stmht[YDIM];         /* storm height [km] from rbinNum[0] */
    static float  clfht[YDIM];         /* clutter-free height [km] from rbinNum[1] */
    static float  flag_rain[YDIM];     /* 0.=no_rain, 1.=possible, 2.=certain */
    static float  flag_PIA[YDIM];      /* 0.<=3 db, 1.>3 db, 2.>10 db */
    static float  flag_rtype[YDIM];    /* 0.=strat, 1.=conv, 2.=others */
    static float  flag_BB[YDIM];       /* 0.=no BB, 1.=BB exist, 2.=warm_rain */
    static float  flag_bottom[YDIM];   /* 0.=normal, 1.>2 km, 2.>4 km, 3.=missing */

/* ------- latent heating algorithm derived variable ------- */ 
    int   iter_max, iter_fail;
    static float  stime[YDIM];         /* scan time --> see the code */
    static float  slat[YDIM];          /* latitude [deg] */
    static float  slon[YDIM];          /* longitude [deg] */
    static float  trop[YDIM];          /* troppause height [km] */
    static float  cldt[YDIM];          /* cloud top height [km] */
    static float  cldb[YDIM];          /* cloud bottom height [km] */
    static float  w[ZDIM][YDIM];       /* vertical wind profile [m/s] */
    static float  fqp[ZDIM][YDIM];     /* production rate of qp [kg/s] */
    static float  lh[ZDIM][YDIM];      /* latent heating profile [K/hr] */
    static float  flag_wtype[YDIM];    /* w-profile type flag
                                          0.=no_rain, 1.=conv, 2.=strat, 3.=anvil, 4.=shallow */
    static float  flag_iter[YDIM];     /* iteration calcuration flag
                                          0.=normal, 1.=failed, 2.=1st guess error,  
                                          3.=2nd guess error, 4.=other error */ 
    static float  cos_antang[YDIM];    /* cosin antenna angle */

/* ------- Output 3D/2D grid variables ("avg" means averaging by each data num) ------- */    
    static float  gze[NZ][NY][NX];          /* 3D avg reflectivity [dBZ] */
    static float  grain[NZ][NY][NX];        /* 3D avg rainfall [mm/hr] */
    static float  dn_rain[NZ][NY][NX];      /* data num of ze and rain >= 0 */
    static float  glh[NZ][NY][NX];          /* 3D avg latent heating [K/day] */
    static float  gw[NZ][NY][NX];           /* 3D avg estimated w-prof [m/s] */
    static float  gfqp[NZ][NY][NX];         /* 3D avg qp-production rate [kg/s] */
    static float  dn_lh[NZ][NY][NX];        /* data num of effective lh, w, fqp */

    static float  gsfcz[NY][NX];            /* avg near-surface reflectivity [dBZ] */
    static float  gsfcr[NY][NX];            /* avg near-surface rainfall [mm/hr] */
    static float  dn_sfcr[NY][NX];          /* data num of sfcz and sfcr >= 0 */
    static float  gsfcht[NY][NX];           /* avg surface height [km] */
    static float  dn_total[NY][NX];         /* total data num */
    static float  dn_iter_error[NY][NX];    /* data num of iteration error */
    static float  gbbht[NY][NX];            /* avg bright band height [km] */
    static float  dn_bbht[NY][NX];          /* data num of bbht */
    static float  gstmht[NY][NX];           /* avg storm height [km] in 2A25 */
    static float  dn_stmht[NY][NX];         /* data num of stmht */
    static float  gcldt[NY][NX];            /* avg cloud top height [km] in PR-LHA */
    static float  dn_cldt[NY][NX];          /* data num of cldt */

    static float  rconv[NY][NX];            /* avg surface rainfall in conv [mm/hr] */
    static float  dn_conv[NY][NX];          /* data num of conv */
    static float  rstrat[NY][NX];           /* avg surface rainfall in strat [mm/hr] */
    static float  dn_strat[NY][NX];         /* data num of strat */
    static float  ranvil[NY][NX];           /* avg surface rainfall in anvil [mm/hr] */
    static float  dn_anvil[NY][NX];         /* data num of anvil */
    static float  rshall[NY][NX];           /* avg surface rainfall in shallow [mm/hr] */
    static float  dn_shall[NY][NX];         /* data num of shall */

    static float  rLST[24][NY][NX];         /* avg surface rainfall in each LST [mm/hr] */
    static float  dn_LST[24][NY][NX];       /* data num of rLST */
    static float  r00UTC[NY][NX];           /* avg surface rainfall in 00-06 UTC [mm/hr] */
    static float  dn_00UTC[NY][NX];         /* data num of r00UTC */
    static float  r06UTC[NY][NX];           /* avg surface rainfall in 06-12 UTC [mm/hr] */
    static float  dn_06UTC[NY][NX];         /* data num of r06UTC */
    static float  r12UTC[NY][NX];           /* avg surface rainfall in 12-18 UTC [mm/hr] */
    static float  dn_12UTC[NY][NX];         /* data num of r12UTC */
    static float  r18UTC[NY][NX];           /* avg surface rainfall in 18-24 UTC [mm/hr] */
    static float  dn_18UTC[NY][NX];         /* data num of r18UTC */

/* ------- GrADS related variables ------- */    
    FILE  *fpg;
    char  g_name[NFNAME];
    int   m, n, mc, nc;
    int   outsize1, outsize2, outsize3;
    char   c_var[400], c_memo[128];
    int    nx, ny, nz, nt, tdim, iyear, iday, imon, ihr, imin;
    double xmin, ymin, zmin, tmin, dx, dy, dz, dt, no_data;


/* ===========================================================================
      OPEN FILES, READ PARAMETERS
   =========================================================================== */
    if(argc < 3){
        printf("Usage: %s {Input files list} {Output file name}\n",argv[0]);
        exit(0);
    }

/* ------- open input files list ------- */
    strcpy(fl_name,argv[1]);
    if((fp = fopen(fl_name,"r" )) == NULL ){
	printf("Input Files List cannot open ! : %s \n",fl_name); 
        exit(1);
    }

/* ------- open output file (GrADS) ------- */
    strcpy(g_name,argv[2]);
    if((fpg = fopen(g_name,"w" )) == NULL ){
        printf("Output file open error!: %s \n",g_name); 
        exit(1);   
    }

/* ------- read parameter file ------ */
       strcpy(init_file, PARAM_FILE);
       status = read_param(init_file, zwc_a_c0, zwc_a_c1, zwc_a_c2, 
                           zwc_b_c0, zwc_b_c1, zwc_b_c2, rho_km, 
                           tempa_km, press_km, vratio_km);
       if(status > 1){
	  if(status<99) printf("data read error: %d mismatched in %s\n", 
                        status, init_file);
          if(status == 99) printf("file open error: %s\n",init_file);
          exit(1);
       } 

/* ------- get air pressure, temparature, qvs, density, and vratio ------- */
       status = air_param(press_km, tempa_km, rho_km, vratio_km, 
                          press, tempa, qvs, rho, vratio);

/* ------- init output variables ------- */    
       for(k=0; k<NZ; k++){
       for(j=0; j<NY; j++){
       for(i=0; i<NX; i++){
         gze[k][j][i]=0.0;
         grain[k][j][i]=0.0;
         dn_rain[k][j][i]=0.0;
         glh[k][j][i]=0.0;
         gw[k][j][i]=0.0;
         gfqp[k][j][i]=0.0;
         dn_lh[k][j][i]=0.0;
       }}}
       for(j=0; j<NY; j++){
       for(i=0; i<NX; i++){
         gsfcz[j][i]=0.0;
         gsfcr[j][i]=0.0;
         dn_sfcr[j][i]=0.0;
         gsfcht[j][i]=0.0;
         dn_total[j][i]=0.0;
         dn_iter_error[j][i]=0.0;
         gbbht[j][i]=0.0;
         dn_bbht[j][i]=0.0;
         gstmht[j][i]=0.0;
         dn_stmht[j][i]=0.0;
         gcldt[j][i]=0.0;
         dn_cldt[j][i]=0.0;
         rconv[j][i]=0.0;
         dn_conv[j][i]=0.0;
         rstrat[j][i]=0.0;
         dn_strat[j][i]=0.0;
         ranvil[j][i]=0.0;
         dn_anvil[j][i]=0.0;
         rshall[j][i]=0.0;
         dn_shall[j][i]=0.0;

         r00UTC[j][i]=0.0;
         dn_00UTC[j][i]=0.0;
         r06UTC[j][i]=0.0;
         dn_06UTC[j][i]=0.0;
         r12UTC[j][i]=0.0;
         dn_12UTC[j][i]=0.0;
         r18UTC[j][i]=0.0;
         dn_18UTC[j][i]=0.0;
       }}
       for(k=0; k<24; k++){
       for(j=0; j<NY; j++){
       for(i=0; i<NX; i++){
         rLST[k][j][i]=0.0;
         dn_LST[k][j][i]=0.0;
       }}}

/* ===========================================================================
      READ GRANULE DATA (Main loop)
   =========================================================================== */
    lp = 0;
    while(1){

      lp++;

/* ------- open 2A25 granule file ------- */
      /*    fgets(f_name,sizeof(f_name),fp);
             sscanf(f_name,"%s",granuleID);  */
      status = fscanf(fp,"%s",f_name);   /* if no more file, status= -1 */
      printf("\n");
      printf("# file num=%3d  file name: %s\n",lp,f_name);
      if(status != 1){
	printf("END of the input files list: %s\n\n",fl_name);
        fclose(fp);
        break;        /* goto the end of while loop */
      }
      strcpy(granuleID,f_name);	
      if(access(granuleID, 000) == -1){
        printf("Input file not found: %s\n",granuleID);
        lp--;
      }else{

       dataType = TK_L2A_25;     /* TK_L2A_25 is defined in IO.h */
       filemode = TK_READ_ONLY;
       status = TKopen(granuleID, dataType, filemode, &granuleHandle2A25);
       if(status  != TK_SUCCESS) {
	  printf("Toolkit file open error!\n");
          exit(1);
       }

/* ------- check metadata  ------ */
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_BEGIN_DATE, &beginDate);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_BEGIN_TIME, &beginTime);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_END_DATE, &endDate);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_END_TIME, &endTime);
       status = TKreadMetadataChar(&granuleHandle2A25,
                        TK_GRANULE_ID, granuleID);
       status = TKreadMetadataChar(&granuleHandle2A25,
                        TK_ALGORITHM_VERSION, algorithmVersion);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_ATTITUDE_MODE, &attitudeMode);
       status = TKreadMetadataInt(&granuleHandle2A25,
                        TK_ORBIT_SIZE, &orbitSize);

       printf("# granule ID    = %s\n", granuleID );
       printf("# algorithm Version = %s\n", algorithmVersion );
       printf("# attitude Mode (0:forward, 1: backward, 2:yaw manuver) = %d\n", attitudeMode );
       printf("# begin time    = %4d/%2d/%2d, %2d:%2d:%2d \n",
                 beginDate.tkyear, beginDate.tkmonth, beginDate.tkday,
                 beginTime.tkhour, beginTime.tkminute, beginTime.tksecond );
       printf("# end time      = %4d/%2d/%2d, %2d:%2d:%2d \n",
                 endDate.tkyear, endDate.tkmonth, endDate.tkday,
                 endTime.tkhour, endTime.tkminute, endTime.tksecond );
       printf("# num of scan   = %d\n",  orbitSize );

/* ------- for GrADS ctl file ------ */
       if(lp==1){
         iyear  = beginDate.tkyear; 
         imon   = beginDate.tkmonth;
         iday   = beginDate.tkday;
       }

/* ------- cos ant angle (common) ------ */
       if(lp==1){
         for(j=0; j<YDIM; j++){
           cos_antang[j] = cos((j-24)*0.71*RAD);
#ifdef DEBUG
           printf("j=%2d  ang=%5.2f  cos_antang=%6.4f\n",
                  j, (j-24)*0.71, cos_antang[j]);
#endif
         }
       }

/* ------- read tropopause height ------ */
       if(lp==1){        /* IF YOU APPLY LONGER TERM DATA over 1-month, CHECK HERE */
         imonth = beginDate.tkmonth;
         strcpy(init_file, TROP_FILE);
         status = read_trop(init_file, imonth, month_trop);
         if(status > 1){
	   if(status<88) printf("data read error: %d mismatched in %s\n", 
                         status, init_file);
           if(status == 88) printf("abnormal month data: %s\n",init_file);
           if(status == 99) printf("file open error: %s\n",init_file);
           exit(1);
         } 
       }




/* ===========================================================================
      READ ACTUAL DATA and DERIVE WC, qp, Vt
   =========================================================================== */
       sscan = 1;                 /* start scan num */
       escan = orbitSize;         /* end scan num */

/* ------- skip scan ------- */
       for(i=1; i<sscan; i++){
         status = TKreadScan(&granuleHandle2A25,&read_data);
         if(status != TK_SUCCESS){
            printf("### Read Error at scan num=%d\n",i);
         }
       }

/* ------- read scan by scan ------- */
       for(i=sscan; i<=escan; i++){      /* scan loop */
         status = TKreadScan(&granuleHandle2A25,&read_data) ;
         if(status != TK_SUCCESS){
            printf("### Read Error at scan num=%d\n",i);
         }else{
	   ii=i-sscan;    /* ii means output scan number, from 0 to escan-sscan */

	 /*--- get scan time and local time etc ---*/ 
         stime[0] = (float)read_data.scanTime;
         stime[1] = (float)beginDate.tkyear;
         stime[2] = (float)beginDate.tkmonth;
         stime[3] = (float)beginDate.tkday;
         stime[4] = (float)((int)(stime[0]/3600.));
         stime[5] = (float)((int)((stime[0]-stime[4]*3600.)/60.));
         stime[6] = (float)((int)(stime[0]-stime[4]*3600.-stime[5]*60.));
         slon_nadir = (float)read_data.geolocation[24][1];
	 stime[10] = stime[0]+(12.0*3600.0*slon_nadir/180.0);   /* local time in sec*/
         stime[11] = stime[1];
         stime[12] = stime[2];
         stime[13] = stime[3];
         if(stime[10]<0.0){
            stime[13] = stime[3]-1.0;
            stime[10] += 24.0*3600.0; 
         }
         if(stime[10]>24.0*3600.0){
            stime[13] = stime[3]+1.0;
            stime[10] -= 24.0*3600.; 
         }
         stime[14] = (float)((int)(stime[10]/3600.));  /* local time in hour */
         stime[15] = (float)((int)((stime[10]-stime[14]*3600.)/60.));
         stime[16] = (float)((int)(stime[10]-stime[14]*3600.-stime[15]*60.));
         for(j=7; j<10; j++) stime[j]=UNDEF;
         for(j=17; j<49; j++) stime[j]=UNDEF;


/* ------- angle bin loop ------- */
         for(j=0; j<NANGLE; j++){
            slat[j]  = (float)read_data.geolocation[j][0];
            slon[j]  = (float)read_data.geolocation[j][1];

            /*--- get rainfall flag in 2A25 ------------------------------
	          0: (bit  1) no rain
	   	 +1: (bit  1) rain possible
	   	 +2: (bit  2) rain certain
	   	 +4: (bit  3) zeta^beta > 0.5 (PIA larger than 3 dB)
	   	 +8: (bit  4) large attenuation (PIA larger than 10 dB)
	        +16: (bit  5) stratiform
	        +32: (bit  6) convective
	        +64: (bit  7) BB exist
	       +128: (bit  8) warm rain
	       +256: (bit  9) rain bottom above 2 km
	       +512: (bit 10) rain bottom above 4 km
	     +16384: (bit 15) data missing between rain top and bottom
	    -------------------------------------------------------------*/
            /*--- output flag as 2A25 original --------------------------
               flag_rain[49]:    0.=no_rain, 1.=possible, 2.=certain
               flag_PIA[49]:     0.<=3 db, 1.>3 db, 2.>10 db
	       flag_rtype[49]:   0.=strat, 1.=conv, 2.=others <<original>>
               flag_BB[49]:      0.=no_rain?, 1.=BB exist, 2.=warm_rain
               flag_rbottom[49]: 0.=normal, 1.>2 km, 2.>4 km, 3.=missing
 	    -------------------------------------------------------------*/
            flag_rain[j]=0.0;  flag_PIA[j]=0.0;  flag_rtype[j]=2.0;  
            flag_BB[j]=0.0;   flag_bottom[j]=0.0;
            if( (read_data.rainFlag[j]&1)      ==1)    flag_rain[j]=1.0;
            if(((read_data.rainFlag[j]&2)>>1)  ==1)    flag_rain[j]=2.0;
            if(((read_data.rainFlag[j]&4)>>2)  ==1)    flag_PIA[j]=1.0;
            if(((read_data.rainFlag[j]&8)>>3)  ==1)    flag_PIA[j]=2.0;
            if(((read_data.rainFlag[j]&16)>>4) ==1)    flag_rtype[j]=0.0;
            if(((read_data.rainFlag[j]&32)>>5) ==1)    flag_rtype[j]=1.0;
            if(((read_data.rainFlag[j]&64)>>6) ==1)    flag_BB[j]=1.0;
            if(((read_data.rainFlag[j]&128)>>7)==1)    flag_BB[j]=2.0;
            if(((read_data.rainFlag[j]&256)>>8)==1)    flag_bottom[j]=1.0;
            if(((read_data.rainFlag[j]&512)>>9)==1)    flag_bottom[j]=2.0;
            if(((read_data.rainFlag[j]&16384)>>14)==1) flag_bottom[j]=3.0;

            /*--- get surface & BB height [km] --------------------------
               range bin numbers of top[0], bottom[1],surface[2], BB[3],
                                    PIZ_threshold[4], ZeMax[5]
	    -------------------------------------------------------------*/
            for(l=0; l<N6; l++){
               rbinNum[l] = (int)read_data.rangeBinNum[j][l];
            }
            rbupTop    = 79-rbinNum[0];     /* Top range bin num upward */
            rbupCLF    = 79-(rbinNum[1]-1); /* Clutter-free range bin num upward */
            rbupSurf   = 79-rbinNum[2];     /* Surf range bin num upward */
            rbupBB     = 79-rbinNum[3];     /* BB range bin num upward */
            /* Memo: - rbinNum[0] is defined as the 1 km above the 
                       rain-certain bin (in ver. 5.53), but it may be 
                       the highest obs range bin if it is no-rain.
	             - rbinNum[1] is the top of the possible surface
                       cluttered range. Hence, "rbinNum[1]-1" is 
                       the actual valid rain bottom data. The near-
                       surface-rain must be the same as rain[rbinNum[1]-1]
            */
            stmht[j] = rbupTop*ZRESO*cos_antang[j];  /* storm top height */
            clfht[j] = rbupCLF*ZRESO*cos_antang[j];  /* clutter-free height */
            sfcht[j] = rbupSurf*ZRESO*cos_antang[j]; /* actual surface height */
            bbht[j]  = rbupBB*ZRESO*cos_antang[j];   /* bright band height */
            cldt[j] = rbupTop*ZRESO*cos_antang[j];   /* temporal(initial) cloud top height */
            cldb[j] = rbupSurf*ZRESO*cos_antang[j];  /* temporal(initial) cloud bottom height */

            if(flag_rain[j] < 1.1){    /* no-rain or rain-possible */
	       cldt[j] = 0.0;
               stmht[j] = 0.0;
            }

            if(bbht[j]>0.0){
               bbht[j] = bbht[j]*(5./6.) - 0.5;   /* bbht is defined as +3 dec.C */
	    }else{
	       bbht[j]=0.0;
            }
/**********************************************************
==> rangeBinNum2r4.c <==    refered from 2A25 USERS GUIDE
Version 5.5 of 2A25 

NOTE:
  The freezing height correction factor (fhcf) is introduced here 
because the freezing level from 2A-23 is a rather conservative (low end) 
estimate in order to avoid a wrong identification of warm rain.  
2A-25 uses the freezing estimate to determine the k-Z and Z-R coefficients.  
For this purpose, it is probably better to over-estimate the freezing height 
than to under-estimate it.  The current version of 2A-23 uses 5 deg.C/km as 
the lapse rate for the conservative estimate.  We change it to 6 deg.C/km
in this program. Hence the correction factor is 6/5=1.2.  
    The freezing height correction facotr (fhcf) is now written in the 
parameter file.  It is read in the main() and transfered into this function 
as an argument. Log: startDist added on 16 Dec. 1997.

====> Since this is special purpose for only 2A25, 
      I should use 2A23 original freezing height.

* 2A23 ver 5.4 uses 6 deg.C/km
* 2A23 defines the freezing height (freezH) as 0 C level
************************************************************/


            /*--- new flag (flag_wtype) to choice the w-profile shape ----
                flag_wtype 
                0.=NO_RAIN, 1.=CONV, 2.=STRAT, 3.=ANVIL, 4.=SHALLOW
 	    -------------------------------------------------------------*/
            /* if(flag_rain[j] > 0.9){                      /* rain possible */
            if(flag_rain[j] > 1.9){                         /* rain certain */
               if(flag_rtype[j]<0.1){                       /* original strat */
                  if(rbupTop>=rbupBB) flag_wtype[j]=2.0;  
                  if(rbupTop<rbupBB)  flag_wtype[j]=4.0;
               }
               if(0.9<flag_rtype[j] && flag_rtype[j]<1.1){  /* original conv */
		  flag_wtype[j]=1.0;
               }
               if(1.9<flag_rtype[j] && flag_rtype[j]<2.1){  /* original others */
		  flag_wtype[j]=3.0;                        /* anvil */
                  if(rbupTop<rbupBB)  flag_wtype[j]=4.0;    /* new flag of shallow */
               }
            }else{
	       flag_wtype[j]=0.0;  /* no_rain */
            }


            /*--- Z-WC relation using 2A25 raintype <not used> ----------
                 (-1=NO_RAIN, 0=STRAT, 1=CONV, 2=OTHERS) 
                 in this routine, anvil => STRAT, shallow => OTHERS
            -------------------------------------------------------------*/
            raintype=-1;     /* including rain possible */
            if(((read_data.rainFlag[j]&16)>>4) ==1) raintype=0; /* 2A25 strat */
            if(((read_data.rainFlag[j]&32)>>5) ==1) raintype=1; /* 2A25 conv */
            if(((read_data.rainFlag[j]&128)>>7)==1) raintype=2; /* 2A23 warm rain */
            if(2.9<flag_wtype[j] && flag_wtype[j]<3.1) raintype=0;  /* anvil */
            if(3.9<flag_wtype[j] && flag_wtype[j]<4.1) raintype=2;  /* shallow */
            /*--- flag_warm (1=warm rain) for Z-WC relation ---*/
            flag_warm =0;
            if(raintype==2) flag_warm=1;

            /*--- Original Z-WC relation using 2A25 raintype --------------
                 (-1=NO_RAIN, 0=STRAT, 1=CONV, 2=OTHERS) 
                 in this routine, anvil => OTHERS, shallow => OTHERS
            -------------------------------------------------------------*/
            raintype=-1;
            if( (read_data.rainFlag[j]&1)      ==1) raintype=0; /* rain possible */ 
            if(((read_data.rainFlag[j]&16)>>4) ==1) raintype=0; /* 2A25 strat */
            if(((read_data.rainFlag[j]&32)>>5) ==1) raintype=1; /* 2A25 conv */
            if(((read_data.rainFlag[j]&128)>>7)==1) raintype=2; /* 2A25 others */
            flag_warm =0;
            if(((read_data.rainFlag[j]&128)>>7)==1) flag_warm=1;
            /* the original warm flag was defined by 1.5 km below the BB */


            /*--- get coefficients of Z-WC relation ---*/
            nubfCF   = read_data.nubfCorrectFactor[j][1];
	    epsilonF = read_data.epsilon[j];
	    weightF  = read_data.weightW[j];
            /*-------------------------------------------------------- 
              If you use 2A25 version 4 product, remove the commentted 
              following 2 lines to use the fixed epsilon.  
                  epsilonF = 1.0;
                  weightF = 0.0;
	      ---------------------------------------------------------*/ 

            /*--- get range bin number of the 5 Node ---*/
            for(l=0; l<N5; l++){
               rbinNode[l] = (int)read_data.ZRParmNode[j][l];
            }

            status = wc_param(rbinNum, raintype, flag_warm, zwc_a_c0, zwc_a_c1, zwc_a_c2, 
                              zwc_b_c0, zwc_b_c1, zwc_b_c2, nubfCF, epsilonF, 
                              weightF, rbinNode, zwc_a, zwc_b);

#ifdef DEBUG
            if(j==24){
               printf("# ii=%4d  nubfCF=%6.4f  epsilonF=%6.4f  weightF=%6.4f",
		      ii, nubfCF, epsilonF, weightF);
               printf("  rbinNode[0]=%2d  [1]=%2d  [2]=%2d  [3]=%2d  [4]=%2d\n",
		      rbinNode[0],rbinNode[1],rbinNode[2],rbinNode[3],rbinNode[4]);
               printf("                                                  ");
               printf("  rbinNum[0]=%2d  [1]=%2d  [2]=%2d  [3]=%2d  [4]=%2d  [5]=%2d\n",
		      rbinNum[0],rbinNum[1],rbinNum[2],rbinNum[3],rbinNum[4],rbinNum[5]);
            }
            if(j==24 && raintype!=-1){ 
               for(k=rbinNum[0]; k<rbinNum[1]; k++){
	         printf("# j=%2d  k=%2d  zwc_a=%10.4f  zwc_b=%10.4f\n",j,k,zwc_a[k],zwc_b[k]);
               }
	    }
#endif

            /*--- get surface Ze & rain ---*/
            if(flag_rain[j] > 1.9){         /* rain certain */
               sfcz[j] = (float)read_data.nearSurfZ[j];
               sfcr[j] = (float)read_data.nearSurfRain[j];
            }else{
               sfcz[j] = UNDEF;
               sfcr[j] = UNDEF;
            }

            /*--- get Ze, Rain, WC, Vt ---------------------------------------*/
	    /* k is downward, which includes read_data.* (2A25), zwc_a, zwc_b */
	    /* kk is upward, which includes rho, vratio, and output variables */
            /* xrain and xze data includes missing value (-99.9) and          */
	    /*                             clutter value (-88.9)              */
            /*----------------------------------------------------------------*/
            for(k=0; k<NRANGE; k++){
               kk=NRANGE-1-k;                          /* the same as 79-k */
               kk = (int)(kk*cos_antang[j]+0.5);       /* Debugged on July 14 */
               xrain = (float)read_data.rain[j][k];             /* [mm/hr] */
               xze  = (float)read_data.correctZFactor[j][k];    /* [dBZ] */
               if(xze < 0.0) xze = 0.0;
               xz = pow(10.0,xze/10.0);                         /* [mm^6/m^3] */
               xwc = zwc_a[k] * pow(xz,zwc_b[k]);               /* [g/m^3] */
               xqp = xwc/rho[kk];                               /* [g/kg] */
               if(xwc > 0.0 && xrain > 0.0){
                  xvt = xrain/(3.6*xwc);                        /* [m/s] */
               }else{
		  xvt = 0.0;
               }
               xvt0 = xvt/vratio[kk];                           /* [m/s] */
               if(xrain >= 0.0){                  /* CHANGE ">" to ">=" on 02July02 */ 
                  cze[kk][j]  = xze;
                  rain[kk][j] = xrain;
                  wc[kk][j] = xwc * 1.0e-3;       /* into [kg/m^3] */
                  qp[kk][j] = xqp * 1.0e-3;       /* into [kg/kg] */
                  vt[kk][j] = xvt * (-1.0);       /* upward is plus */
                  vt0[kk][j] = xvt0 * (-1.0);     /* upward is plus */
               }else{
		  cze[kk][j]  = UNDEF;   /* clutter data (xrain=-88.9) is UNDEF */
                  rain[kk][j] = UNDEF;
                  wc[kk][j] = UNDEF;
                  qp[kk][j] = UNDEF;
                  vt[kk][j] = UNDEF;
                  vt0[kk][j] = UNDEF;
               }
	    }

            /*--- print scan num, date, and time ---*/
            if(first_scan_sec==0.0) first_scan_sec = stime[0];
            if(j==24 && i==sscan){       /* on nadir */
               printf("First scan: %4d (t=%4d)  %4d/%2d/%2d,%2d:%2d:%2d ",
		       i,i-sscan,(int)stime[1],(int)stime[2],(int)stime[3],
		                 (int)stime[4],(int)stime[5],(int)stime[6]);
	       printf("(%2d,%2d:%2d:%2d LST) lat=%7.3f lon=%8.3f (at nadir)\n",
                       (int)stime[13],(int)stime[14],(int)stime[15],(int)stime[16],
                       slat[j],slon[j]);
            }
            if(j==24 && i==escan){       /* on nadir */
               printf("Last scan : %4d (t=%4d)  %4d/%2d/%2d,%2d:%2d:%2d ",
		       i,i-sscan,(int)stime[1],(int)stime[2],(int)stime[3],
		                 (int)stime[4],(int)stime[5],(int)stime[6]);
	       printf("(%2d,%2d:%2d:%2d LST) lat=%7.3f lon=%8.3f (at nadir)\n",
                       (int)stime[13],(int)stime[14],(int)stime[15],(int)stime[16],
                       slat[j],slon[j]);
            }

         }  /* end of angle loop */
         }  /* if-else of readScan error */

/* ===========================================================================
      GET CLOUD TOP and BOTTOM HEIGHTS
   =========================================================================== */
         /*--- get cloud top and bottom height ---*/
         status = cld_ht(rain, cos_antang, flag_wtype, slat,
                         month_trop, trop, cldt, cldb);

         for(j=0; j<YDIM; j++){
            /* eliminate abnormal surface height [km] */ 
            if(sfcht[j]>=cldt[j] || sfcht[j]>8.848){
               sfcht[j]=0.0;
               bbht[j]=0.0;
               cldb[j]=0.0;
               flag_wtype[j]=0.0;
            }

            /* eliminate abnormal cldt above trop (almost noise) */
            /*
            if(cldt[j]>19.5) cldt[j]=19.5;
            if(cldt[j]<0.0)  cldt[j]=0.0;
            if(cldb[j]>=trop[j])  cldb[j]=trop[j]-ZRESO;
            if(cldb[j]<=sfcht[j]) cldb[j]=sfcht[j]+ZRESO;
            */

            if(cldb[j]>=trop[j]){
#ifdef DEBUG
	       printf("#==[cldb>trop]== j=%2d wtype=%2.1f trop=%5.2f cldt=%5.2f ",
                       j,flag_wtype[j],trop[j],cldt[j]);
	       printf("bbht=%5.2f cldb=%5.2f sfcht=%5.2f\n",bbht[j],cldb[j],sfcht[j]);
#endif
	       cldt[j]=0.0;
	       cldb[j]=0.0;
               flag_wtype[j]=0.0;
            }

            if(cldt[j]>=trop[j]){
#ifdef DEBUG
	       printf("#==[cldt>trop]== j=%2d wtype=%2.1f trop=%5.2f cldt=%5.2f ",
                       j,flag_wtype[j],trop[j],cldt[j]);
	       printf("bbht=%5.2f cldb=%5.2f sfcht=%5.2f\n",
                       bbht[j],cldb[j],sfcht[j]);
#endif
               if(0.9<flag_wtype[j] && flag_wtype[j]<1.1){  /* conv */
                 trop[j]=cldt[j]+ZRESO;
               }
               if(1.9<flag_wtype[j] && flag_wtype[j]<3.1){  /* strat or anvil */
                  if((cldt[j]-trop[j])<0.5){
                     trop[j]=cldt[j]+ZRESO;
                  }else{
	             cldt[j]=0.0;
	             cldb[j]=0.0;
                     flag_wtype[j]=0.0;
                  }
               }
               if(3.9<flag_wtype[j] && flag_wtype[j]<4.1){  /* shallow */
	          cldt[j]=0.0;
	          cldb[j]=0.0;
                  flag_wtype[j]=0.0;
               }

            }

         }

/* ===========================================================================
      ELIMINATE UNNECESSARY qp and AVERAGE qp and vt
   =========================================================================== */
         for(j=0; j<YDIM; j++){
	   kcldt = cldt[j]/ZRESO;    /* 0 <= kcldt <= 79 */
	   kcldb = cldb[j]/ZRESO;
           /*--- eliminate qp above the cldt, under the cldb---*/
           for(k=0; k<kcldb+1; k++){
             qp[k][j]=UNDEF;
           }
           for(k=kcldt; k<ZDIM; k++){
             qp[k][j]=UNDEF;
           }

           /*--- eliminate BB of qp using the 5 bins average of qp ---*/
           for(k=0; k<ZDIM; k++){
             xxqp[k]=qp[k][j];
           }
           for(k=kcldb+2; k<kcldt-2; k++){
             qpsum=0.0;  qpnum=0.0;
             for(kk=k-2; kk<k+3; kk++){
               if(xxqp[kk]>=0.0){
                 qpsum += xxqp[kk];
                 qpnum += 1.0;
               }
             }
             if(qpnum>1.0){
               qpdiff = (xxqp[k]-(qpsum/qpnum))*1000.0;  /* [g/kg] */
               if(qpdiff>0.2){
                 qp[k][j]=qpsum/qpnum;
#ifdef DEBUG3
                 printf("## j=%2d  k=%2d  xxqp=%f  avg=%f  diff=%f\n",
			j, k, xxqp[k], qpsum/qpnum, qpdiff);
#endif
               }
             }
           }

           /*--- 5 bins averaging of rain, qp and vt ---*/
           for(k=0; k<ZDIM; k++){
             xxrain[k]=rain[k][j];
             xxqp[k]=qp[k][j];
             xxvt[k]=vt[k][j];
           }
           for(k=kcldb+2; k<kcldt-2; k++){
             rainsum=0.0;  rainnum=0.0;
             qpsum=0.0;    qpnum=0.0;
             vtsum=0.0;    vtnum=0.0;
             for(kk=k-2; kk<k+3; kk++){
               if(xxrain[kk]>=0.0 && xxqp[kk]>=0.0){
                 rainsum += xxrain[kk];
                 rainnum += 1.0;
                 qpsum += xxqp[kk];
                 qpnum += 1.0;
               }
               if(xxvt[kk]>UNDEF){
                 vtsum += xxvt[kk];
                 vtnum += 1.0;
               }
             }
             if(qpnum>1.0) qp[k][j]=qpsum/qpnum;
             if(rainnum>1.0) vt[k][j]=-1.0*(rainsum/rainnum)/(3600.0*qp[k][j]*rho[k]);
             /* if(vtnum>1.0) vt[k][j]=vtsum/vtnum; */
#ifdef DEBUG3
             printf("## j=%2d  k=%2d  qp=%f %f  qpsum=%f %f  vt=%f %f  vtsum=%f %f\n",
                     j, k, qp[kk][j], xxqp[kk], qpsum, qpnum, 
		    vt[kk][j], xxvt[kk], vtsum, vtnum);
#endif
           }

           /*--- fill lowest/2nd-lowest qp and vt ---*/
           lowest_flag=0;
           for(k=0; k<ZDIM-3; k++){
              if(lowest_flag==0 &&
                 qp[k+2][j]>=0.0 && qp[k+3][j]>=0.0){
	            qp[k+1][j] = qp[k+3][j] + 2.0*(qp[k+2][j]-qp[k+3][j]);
	            qp[k][j] =   qp[k+3][j] + 3.0*(qp[k+2][j]-qp[k+3][j]);
	            vt[k+1][j] = vt[k+3][j] + 2.0*(vt[k+2][j]-vt[k+3][j]);
	            vt[k][j] =   vt[k+3][j] + 3.0*(vt[k+2][j]-vt[k+3][j]);
                    lowest_flag=1;
              }
           } 

         }  /* end of j-loop (j<YDIM) */

/* ===========================================================================
      RETRIEVE w-, Fqp-, LH-profile
   =========================================================================== */
         /*--- retrieve LH profile ---*/
         iter_max = lh_prof(rho, qvs, cos_antang, qp, vt, vt0, sfcr, 
		  	        sfcht, bbht, trop, cldt, cldb,  
                                flag_wtype, w, fqp, lh, flag_iter);

         iter_fail=0;  /* number of failures in iteration (0-49) */
         for(j=0; j<YDIM; j++){
           if(flag_iter[j]>0.0) iter_fail +=1;
         }
         if(iter_fail>0){
           printf("------ %2d times failures occured (max iter=%2d) in the scan: ",
                   iter_fail,iter_max);
           printf("%4d (t=%4d)  %2d:%2d:%2d  lat=%7.3f lon=%8.3f ------\n",
                   i,i-sscan,(int)stime[4],(int)stime[5],(int)stime[6],slat[24],slon[24]);
         }

/* ===========================================================================
      ACCUMULATE DATA in each grid
   =========================================================================== */
         /*--- reset UNDEF values ---*/
         for(j=0; j<YDIM; j++){
            if(flag_rain[j] < 1.1){ /* no-rain or rain-possible */
	       stmht[j] = UNDEF;
               bbht[j] = UNDEF;
	       cldt[j] = UNDEF;
               cldb[j] = UNDEF;
            }
         }

         /*--- get grid point ---*/
         for(j=0; j<YDIM; j++){
	   nx = (int)((slon[j]-LON1)/XGRID);
	   ny = (int)((slat[j]-LAT1)/YGRID);
           if(0<=nx && nx<NX && 0<=ny && ny<NY){
             /* accumulate 2D data */
             gsfcht[ny][nx] += sfcht[j];
             dn_total[ny][nx] += 1.0;
             /* surface rain is not UNDEF */
             if(sfcr[j]>=0.0){
               gsfcz[ny][nx] += pow(10.0,sfcz[j]*0.1);
	       gsfcr[ny][nx] += sfcr[j]; 
               dn_sfcr[ny][nx] += 1.0;

               /* new rain (w-prof) type */
               if(flag_wtype[j]==1){
                 rconv[ny][nx] += sfcr[j];
                 dn_conv[ny][nx] += 1.0;
               }
               if(flag_wtype[j]==2){
                 rstrat[ny][nx] += sfcr[j];
                 dn_strat[ny][nx] += 1.0;
               }
               if(flag_wtype[j]==3){
                 ranvil[ny][nx] += sfcr[j];
                 dn_anvil[ny][nx] += 1.0;
               }
               if(flag_wtype[j]==4){
                 rshall[ny][nx] += sfcr[j];
                 dn_shall[ny][nx] += 1.0;
               }

               /* local time data */
                 nt=(int)(stime[14]+0.5);
                 rLST[nt][ny][nx]  += sfcr[j];
	         dn_LST[nt][ny][nx] += 1.0;

               /* UTC time data */
               if(0.0<=stime[4] && stime[4]<6.0){
                 r00UTC[ny][nx]  += sfcr[j];
	         dn_00UTC[ny][nx] += 1.0;
               }
               if(6.0<=stime[4] && stime[4]<12.0){
                 r06UTC[ny][nx]  += sfcr[j];
	         dn_06UTC[ny][nx] += 1.0;
               }
               if(12.0<=stime[4] && stime[4]<18.0){
                 r12UTC[ny][nx]  += sfcr[j];
	         dn_12UTC[ny][nx] += 1.0;
               }
               if(18.0<=stime[4] && stime[4]<24.0){
                 r18UTC[ny][nx]  += sfcr[j];
	         dn_18UTC[ny][nx] += 1.0;
               }
             } 

             /* other heights data */
             if(bbht[j]>0.0){
               gbbht[ny][nx] += bbht[j];
               dn_bbht[ny][nx] += 1.0;
             }
             if(stmht[j]>0.0){
               gstmht[ny][nx] += stmht[j]; 
               dn_stmht[ny][nx] += 1.0;
             }
             if(cldt[j]>0.0){
               gcldt[ny][nx] += cldt[j];
               dn_cldt[ny][nx] += 1.0;
             }

             /* accumulate 3D data (without iter failed data) */
             if(flag_iter[j]>0.0){
               dn_iter_error[ny][nx] += 1.0;
             }else{
               for(k=0; k<ZDIM; k++){
	         nz = (int)((k*ZRESO*cos_antang[j])/ZGRID);  /* debugged on Aug 7 */
                 if(0<=nz && nz<NZ){
                   if(rain[k][j]>=0.0){
                     gze[nz][ny][nx] += pow(10.0,cze[k][j]*0.1);
                     grain[nz][ny][nx] += rain[k][j];
                     dn_rain[nz][ny][nx] += 1.0;
                   }
                   if(lh[k][j]>UNDEF2){
                     glh[nz][ny][nx] += lh[k][j];
                     gw[nz][ny][nx] += w[k][j];
                     gfqp[nz][ny][nx] += fqp[k][j];
                     dn_lh[nz][ny][nx] += 1.0;
                   }
                 }
	       }
	     }

	   } /* end of if(0<=nx<NX && 0<=ny<NY) */
	 }   /* end of for(j=0; j<YDIM; j++) */

       }    /* end of scan loop */

       /*--- close HDF file ---*/
       status = TKclose(&granuleHandle2A25);

     }  /* end of the DATA exist --- if(access(granuleID, 000) != -1){ */

    } /* enf of while(1) */


/* ===========================================================================
      AVERAGING
   =========================================================================== */
    for(j=0; j<NY; j++){
    for(i=0; i<NX; i++){
       if(dn_total[j][i]>0.0){
         gsfcht[j][i] /= dn_total[j][i];
       }else{
         gsfcht[j][i] = UNDEF;
       }
       if(dn_sfcr[j][i]>0.0){
         gsfcz[j][i] = 10.0*log10(gsfcz[j][i]/dn_sfcr[j][i]);
         gsfcr[j][i] /= dn_sfcr[j][i];
       }else{
         gsfcz[j][i] = UNDEF;
         gsfcr[j][i] = UNDEF;
       }

       if(dn_conv[j][i]>0.0){
         rconv[j][i] /= dn_conv[j][i];
       }else{
         rconv[j][i] = UNDEF;
       }
       if(dn_strat[j][i]>0.0){
         rstrat[j][i] /= dn_strat[j][i];
       }else{
         rstrat[j][i] = UNDEF;
       }
       if(dn_anvil[j][i]>0.0){
         ranvil[j][i] /= dn_anvil[j][i];
       }else{
         ranvil[j][i] = UNDEF;
       }
       if(dn_shall[j][i]>0.0){
         rshall[j][i] /= dn_shall[j][i];
       }else{
         rshall[j][i] = UNDEF;
       }

       for(k=0; k<24; k++){
         if(dn_LST[k][j][i]>0.0){
           rLST[k][j][i] /= dn_LST[k][j][i];
         }else{
           rLST[k][j][i] = UNDEF;
         }
       }

       if(dn_00UTC[j][i]>0.0){
         r00UTC[j][i] /= dn_00UTC[j][i];
       }else{
         r00UTC[j][i] = UNDEF;
       }
       if(dn_06UTC[j][i]>0.0){
         r06UTC[j][i] /= dn_06UTC[j][i];
       }else{
         r06UTC[j][i] = UNDEF;
       }
       if(dn_12UTC[j][i]>0.0){
         r12UTC[j][i] /= dn_12UTC[j][i];
       }else{
         r12UTC[j][i] = UNDEF;
       }
       if(dn_18UTC[j][i]>0.0){
         r18UTC[j][i] /= dn_18UTC[j][i];
       }else{
         r18UTC[j][i] = UNDEF;
       }

       if(dn_bbht[j][i]>0.0){
         gbbht[j][i] /= dn_bbht[j][i];
       }else{
         gbbht[j][i] = UNDEF;
       }
       if(dn_stmht[j][i]>0.0){
         gstmht[j][i] /= dn_stmht[j][i];
       }else{
         gstmht[j][i] = UNDEF;
       }
       if(dn_cldt[j][i]>0.0){
         gcldt[j][i] /= dn_cldt[j][i];
       }else{
         gcldt[j][i] = UNDEF;
       }

       for(k=0; k<NZ; k++){
         if(dn_rain[k][j][i]>0.0){
           gze[k][j][i] = 10.0*log10(gze[k][j][i]/dn_rain[k][j][i]);
           grain[k][j][i] /= dn_rain[k][j][i]; 
         }else{
           gze[k][j][i] = UNDEF;
           grain[k][j][i] = UNDEF;
         }

         if(dn_lh[k][j][i]>0.0){
           glh[k][j][i] /= dn_lh[k][j][i];
           gw[k][j][i] /= dn_lh[k][j][i];
           gfqp[k][j][i] /= dn_lh[k][j][i];
         }else{
           glh[k][j][i] = UNDEF;
           gw[k][j][i] = UNDEF;
           gfqp[k][j][i] = UNDEF;
         }
       }
    }}      

/* ===========================================================================
      OUTPUT DATA
   =========================================================================== */
/*----- output GrADS file ------*/
    outsize1 = NX * NY * sizeof(float);
    outsize2 = NX * NY * NZ * sizeof(float);
    outsize3 = NX * NY * 24 * sizeof(float);
    fwrite(gze,1,outsize2,fpg);
    fwrite(grain,1,outsize2,fpg);
    fwrite(dn_rain,1,outsize2,fpg);
    fwrite(glh,1,outsize2,fpg);
    fwrite(gw,1,outsize2,fpg);
    fwrite(gfqp,1,outsize2,fpg);
    fwrite(dn_lh,1,outsize2,fpg);

    fwrite(gsfcz,1,outsize1,fpg);
    fwrite(gsfcr,1,outsize1,fpg);
    fwrite(dn_sfcr,1,outsize1,fpg);
    fwrite(gsfcht,1,outsize1,fpg);
    fwrite(dn_total,1,outsize1,fpg);
    fwrite(dn_iter_error,1,outsize1,fpg);
    fwrite(gbbht,1,outsize1,fpg);
    fwrite(dn_bbht,1,outsize1,fpg);
    fwrite(gstmht,1,outsize1,fpg);
    fwrite(dn_stmht,1,outsize1,fpg);
    fwrite(gcldt,1,outsize1,fpg);
    fwrite(dn_cldt,1,outsize1,fpg);

    fwrite(rconv,1,outsize1,fpg);
    fwrite(dn_conv,1,outsize1,fpg);
    fwrite(rstrat,1,outsize1,fpg);
    fwrite(dn_strat,1,outsize1,fpg);
    fwrite(ranvil,1,outsize1,fpg);
    fwrite(dn_anvil,1,outsize1,fpg);
    fwrite(rshall,1,outsize1,fpg);
    fwrite(dn_shall,1,outsize1,fpg);

    fwrite(rLST,1,outsize3,fpg);
    fwrite(dn_LST,1,outsize3,fpg);
    fwrite(r00UTC,1,outsize1,fpg);
    fwrite(dn_00UTC,1,outsize1,fpg);
    fwrite(r06UTC,1,outsize1,fpg);
    fwrite(dn_06UTC,1,outsize1,fpg);
    fwrite(r12UTC,1,outsize1,fpg);
    fwrite(dn_12UTC,1,outsize1,fpg);
    fwrite(r18UTC,1,outsize1,fpg);
    fwrite(dn_18UTC,1,outsize1,fpg);

/*----- close the output file -----*/
       fclose(fpg);
       printf("\n");
       printf("output data: %d(lon) * %d(lat) * %d(height) * 4 bytes\n",
               NX,NY,NZ);
       printf("             3D(7) :ze,rain,dn_rain,lh,w,fqp,dn_lh\n");
       printf("             2D(32+48):sfcz,sfcr,dn_sfcr,sfcht,dn_total,dn_iter_error,bbht,dn_bbht,stmht,dn_stmht,cldt,dn_cldt,conv,dn_conv,strat,dn_strat,anvil,dn_anvil,shall,dn_shall,rLST[24],dn_rLST[24],r00,dn_r00,r06,dn_r06,r12,dn_r12,r18,dn_r18\n");
       printf("\n");

/*----- output GrADS ctl file -----*/
        strcpy(c_memo,"3A25 Latent Heating Profile");
        strcpy(c_var,"ze,rain,dn_rain,lh,w,fqp,dn_lh,sfcz,sfcr,dn_sfcr,sfcht,dn_total,dn_iter_error,bbht,dn_bbht,stmht,dn_stmht,cldt,dn_cldt,conv,dn_conv,strat,dn_strat,anvil,dn_anvil,shall,dn_shall,rLST,dn_rLST,r00,dn_r00,r06,dn_r06,r12,dn_r12,r18,dn_r18");

        nx=NX;         ny=NY;       nz=NZ;      nt=1;
        xmin=LON1+XGRID*0.5;     
        ymin=LAT1+YGRID*0.5;   
        zmin=0.0;   
        tmin=0.0;
        dx=XGRID;      dy=YGRID;    dz=ZGRID;   dt=1.0;
        no_data= UNDEF;
        ihr    = 0;
        imin   = 0;

        ggrads_ctl( g_name, nx, ny, nz, nt, xmin, ymin, zmin, tmin,
                    dx, dy, dz, dt, no_data, ihr, imin, iday, imon, iyear,
                    c_var, c_memo );


	return(0);
}
