/**********************************************************
                                 Aug 07, 2002  Shin Satoh
FUNCTION: int lh_prof()
SOURCE FILE: lh_prof.c
PURPOSE: Retrive the Latent Heating Profile

CALLING SEQUENCE:
    (I) float *rho         : air density in range bins [kg/m^3]
    (I) float *qvs         : saturated mixing ratio of vapar in range bins
    (I) float *cos_antang  : cosin antenna angle
    (I) float qp[][49]     : mixing ratio of precipitation [kg/kg]
    (I) float vt[][49]     : terminal falling velocity of precipitation [m/s]
    (I) float vt0[][49]    : air-density non-corrected vt [m/s]
    (I) float *sfcr        : near surface rainfall rate [mm/hr]
    (I) float *sfcht       : surface height [km]
    (I) float *bbht        : bright band height [km]
    (I) float *trop        : tropopause height [km]
    (I) float *cldt        : cloud top height [km]
    (I) float *cldb        : cloud bottom height [km]
    (I) float *flag_wtype  : w-profile type flag
                             (0.=no_rain, 1.=conv, 2.=strat, 3.=anvil, 4.=shallow)
    (O) float w[][49]      : estimated vertical wind (w) [m/s]
    (O) float fqp[][49]    : production rate of precipitaion [kg/s]
    (O) float lh[][49]     : calculated latent heating [K/hr]
    (O) float *flag_iter   : iteration flag (0.=normal, 1.=failed, 2.=1st guess error,  
                                             3.=2nd guess error, 4.=other error)

DEBUG:
  DEBUG1:  iteration problems
  DEBUG2:  profiles of some variables

RETURN: max iteration number, but 99 means fatal error
************************************************************/
#include <math.h>
#include <stdio.h>

#define  YDIM        49      /* number of angle bins in output */
#define  ZDIM        80      /* number of vertical bins in output */
#define  ZRESO        0.25   /* vertical resolution in km */
#define  UNDEF   -9.e+33    /* missing data in output */
#define  UNDEF2  -8.9e+33   /* missing data avoiding roundoff error */

#define  SNOW_THRES  -1.8    /* threshold of vt0 to estimate snow region */
#define  LH_THRES     0.5    /* threshold of diff between srfR & integLH */
#define  ITMAX       30      /* maximum number of iteration (>22) */


int lh_prof (float *rho, float *qvs, float *cos_antang,
             float qp[][YDIM], float vt[][YDIM], float vt0[][YDIM],
             float *sfcr, float *sfcht, float *bbht, 
             float *trop, float *cldt, float *cldb, float *flag_wtype, 
             float w[][YDIM], float fqp[][YDIM], float lh[][YDIM],
             float *flag_iter)
{
      int    i, j, jj, k, it, it2;
      int    iterN[YDIM], iterN_max=0;
      int    wtype, lowest_flag, status;
      float  xvt0, wmax, pmotion;
      float  advqp, advqph, srcqp, dqvsdz, xevacon;
      double zkm, zt, zc, zm;
      double xtrop, xtempC, xes, xpress;

      float  LHsfcR, LHinteg, LHintegS, LHintegN;
      float  wtmp, minsfcR, multipR, guessR, guessLH, satLH;
      float  adjw, adjw0, adjw1, adjadv;
      float  difLH, difLH0, difLH1;

      static float  coef[5], coef0[5];   /* coefficients of w-prof */
      static float  clv=2.50e+6;         /* [J kg^-1] in 0 C */
      static float  clf=0.334e+6;        /* [J kg^-1] in 0 C */
      static float  cp=1004.0;           /* [J K^-1 kg^-1]  for dry air */
      static float  rhol=1000.0;         /* [kg m^-3] density of water */
      static float  dzm=ZRESO*1000.0;    /* [m] */


/*======================================================================*/
/* ------ init output values ------ */
      for(j=0; j<YDIM; j++){
        for(k=0; k<ZDIM; k++){
          w[k][j]=UNDEF;
          fqp[k][j]=UNDEF;
	  lh[k][j]=UNDEF;    
        }
      }

/* ------ angle bin loop ------ */
      for(j=0; j<YDIM; j++){
        
        flag_iter[j]=0.0;
        iterN[j]=0;

      /*--- w-profile type (flag_wtype) ---------------------- 
      (0)no_rain, (1)conv, (2)strat, (3)anvil, (4)shallow
       --------------------------------------------------------*/
        wtype = (int)(flag_wtype[j]+0.5);

      if(wtype>0){     /* Nothing to do in no_rain */

        /* Get initial coefficients of w-profile 
           zt, zc, zm are heights from the surface */
        zt = trop[j]-sfcht[j];
        zc = cldt[j]-sfcht[j];
        if(wtype==1) zm = 0.01;                     /* conv */
        if(wtype==2){                               /* strat */
           zm = bbht[j]-sfcht[j];
           if(zm<0.25) zm = 0.25;        /* bbht < sfcht (mountains) */
           if(zm>(zc*0.5)) zm = zc*0.5;  /* short strat */
        }
        if(wtype==3) zm = cldb[j]+0.25-sfcht[j];    /* anvil */
        if(wtype==4) zm = 0.01;                     /* shallow */

        /* Check the 3 nodes */
        if(wtype==1 || wtype==4){   /* conv */
          if(zm<0.01)        zm = 0.01;
          if((zc-zm)<0.25)   zc = zm+0.25;
          if((zt-zc)<0.25)   zt = zc+0.25;
        }else{                      /* strat or anvil */
          if((zt-zc)<0.25)   zc = zt-0.25;
          if((zc-zm)<0.25)   zm = zc-0.25;
          if(zm<0.01){
	    printf("##### Abnormal heights: zt=%f zc=%f zm=%f ",zt,zc,zm);
            printf(" at j=%2d  wtype=%d  sfcht=%f bbht=%f\n",j,wtype,sfcht[j],bbht[j]);
            zm = 0.01;  zc=0.25;   /* never occure I believe */
          }
        }

	 /* Make qvs profile using the trop and BB heights */
         for(k=0; k<ZDIM; k++){
           zkm=k*ZRESO*cos_antang[j];
           if(bbht[j]<=0.0) printf("##### Abnormal BB height =%f km\n",bbht[j]);
           xtrop = zt+sfcht[j];
           if(zkm<xtrop){
             if(zkm>bbht[j]){                 
               xtempC=3.0-(zkm-bbht[j])*6.0;  /* 3.0 (deg.C) means mean BB height */  
             }else{                           /* 6.0 (deg.C/km) means lapse rate */
               xtempC=3.0-(zkm-bbht[j])*5.0;  /* to avoid largee qvs in lower layers */
             }
           }else{
             xtempC=3.0-(xtrop-bbht[j])*6.0;  /* constant temp. above tropopause */
           }
           xes=6.1078*pow(10.0,(7.5*xtempC/(237.3+xtempC)));
           xpress = rho[k]*287.0*(xtempC+273.15)*0.01;   /* in hPa */      
           qvs[k]=0.622*xes/(xpress-0.378*xes);

#ifdef DEBUG
           printf("  j=%2d  k=%2d  z=%5.2f temp=%6.2f press=%6.1f qvs=%7.4f\n",
                  j,k,zkm,xtempC,xpress,qvs[k]*1000.);
#endif
         }

        /* Gauss-Jordan method to solve the 4th power func */
        status = w4pf_prof(zt, zc, zm, coef0);
        if(status!=0){
	   printf("### Error of 4th power function calc: status=%d\n",status);
           printf("### j=%d wtype=%d trop-sfc=%5.2f cldt-sfc=%5.2f melt/bot-sfc=%5.2f\n",
                       j,wtype,zt,zc,zm);
           printf("### a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f\n",
		       coef0[1],coef0[2],coef0[3],coef0[4]);   /* coef0[1]=1.0 */
           for(jj=0; jj<YDIM; jj++){ flag_iter[jj]=4.0; }
           return 99;  /* never occure I believe */
        }

        /* Saturation LH integration for 1st guess -------------
         satLH = -(Lv/Cp) * w * (dqvs/dz)     : satLH --[K/s]
         Integ[ rho*LH ]dz = (Lv/Cp) * Rsfc   : Rsfc --[mm/s]
        -------------------------------------------------------*/
        LHintegS = 0.0;
        for(k=1; k<ZDIM-1; k++){
	  zkm = k*ZRESO*cos_antang[j] - sfcht[j];        /* height [km] from the surface */
          dqvsdz = (qvs[k+1]-qvs[k-1])/(dzm*2.0);        /* dqvs/dz */
          wtmp = pow(zkm,4.) + coef0[2]*pow(zkm,3.) +
	         coef0[3]*pow(zkm,2.) + coef0[4]*zkm;    /* temporary w [m/s] */ 
          xevacon=-1.0 * wtmp * dqvsdz;
          xvt0=vt0[k][j];
          if(xvt0>SNOW_THRES){  /* snow region */
             satLH=((clv+clf)/cp)*xevacon*3600.0;        /* [K/hr] */
          }else{
             satLH=(clv/cp)*xevacon*3600.0;
          }

          /* conv or shallow */
          if((wtype==1 || wtype==4) && 0.0<=zkm && zkm<=zc){
             LHintegS += rho[k] * satLH/3600.0 * dzm;
          }
          /* strat or anvil */
          if((wtype==2 || wtype==3) && zm<=zkm && zkm<=zc){
             LHintegS += rho[k] * satLH/3600.0 * dzm;
          }
        }
 
#ifdef DEBUG
        /* check LHintegS */
        if(LHintegS < 0.0){
            printf("####### abnormal LHintegS at j=%2d wtype=%1d ",
                   j,wtype);
            printf("sfcR=%7.3f sfcHT=%5.2f : LHintegS=%f  ",
	           sfcr[j],sfcht[j],LHintegS);
            printf("w=0 at %5.2f, %5.2f,%5.2f,%5.2f\n",
                   zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
            LHintegS = 0.0;
            for (k=1; k<ZDIM-1; k++) {
	      zkm = k*ZRESO*cos_antang[j] - sfcht[j];
              dqvsdz = (qvs[k+1]-qvs[k-1])/(dzm*2.0);
              wtmp = pow(zkm,4.) + coef0[2]*pow(zkm,3.) +
	             coef0[3]*pow(zkm,2.) + coef0[4]*zkm;
              xevacon=-1.0 * wtmp * dqvsdz;
              xvt0=vt0[k][j];
              if(xvt0>SNOW_THRES){
                satLH=((clv+clf)/cp)*xevacon*3600.0;
              }else{
                satLH=(clv/cp)*xevacon*3600.0;
              }
              if((wtype==1 || wtype==4) && 0.0<=zkm && zkm<=zc){
                LHintegS += rho[k] * satLH/3600.0 * dzm;
                printf("   %2d %5.2f km: w=%f evacon=%f LHintegS=%f qvs=%f qp=%f\n",
                       k,zkm,wtmp,xevacon*1e3,LHintegS*1e3,qvs[k]*1e3,qp[k][j]*1e3);
              }
              if((wtype==2 || wtype==3) && zm<=zkm && zkm<=zc){
                LHintegS += rho[k] * satLH/3600.0 * dzm;
                printf("   %2d %5.2f km: w=%f evacon=%f LHintegS=%f qvs=%f qp=%f\n",
                       k,zkm,wtmp,xevacon*1e3,LHintegS*1e3,qvs[k]*1e3,qp[k][j]*1e3);
              }
	    }
        }
#endif

        /* coefficients for iteration in each raintype */
        if(wtype==1){     /* conv */
	   multipR = 5.0;
           minsfcR = 0.5;
        }
        if(wtype==2){     /* strat */
	   multipR = 5.0;
           minsfcR = 0.5;
        }
        if(wtype==3){     /* anvil */
	   multipR = 2.0;
           minsfcR = 0.5;
        }
        if(wtype==4){     /* shallow */
	   multipR = 5.0;
           minsfcR = 0.5;
        }

        /* make the first guess (smaller LHinteg) of the w-profile */
        if(sfcr[j]<minsfcR){          /* anvil or very weak sfcR */
	   guessR = minsfcR;          /* [mm/hr] */
        }else{
	   guessR = sfcr[j];
        }
        guessLH = (clv/cp)*guessR/3600.; 
        adjw0=guessLH/LHintegS;     /* adjustment of w-profile max */



#ifdef DEBUG1
        printf("\n");
        printf("# j=%2d wtype=%d trop=%5.2f cldt=%5.2f melt=%5.2f sfcR=%5.1f\n",
                j,wtype,zt,zc,zm,sfcr[j]);
        if(adjw<0){
	  printf("### j=%2d  adjw0=%f  guessR=%f  LHintegS=%f\n",
		 j,adjw0,guessR,LHintegS);
        }
        coef[1]=adjw0;
        coef[2]=coef0[2]*coef[1];
        coef[3]=coef0[3]*coef[1];
        coef[4]=coef0[4]*coef[1];
        wmax = 0.0;
        for (k=0; k<ZDIM; k++) {
	  zkm = k*ZRESO*cos_antang[j] - sfcht[j];
          w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	            coef[3]*pow(zkm,2.) + coef[4]*zkm;
          if(zkm < 0.0) w[k][j]=0.0;
          if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
          if(wmax < w[k][j]) wmax=w[k][j];
        }
        printf("# 1st guess: a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f  |  wmax=%6.2f\n",
                coef[1],coef[2],coef[3],coef[4],wmax);
#endif

        /* make the initial second guess (larger LHinteg ) of the w-profile */
        if(sfcr[j]<minsfcR){
           guessR = minsfcR * multipR;
        }else{
           guessR = sfcr[j] * multipR;
        }
	if(guessR>300.0) guessR = 300.0;  /* [mm/hr] */
        guessLH = (clv/cp)*guessR/3600.; 
        adjw1=guessLH/LHintegS;     /* adjustment of w-profile max */

#ifdef DEBUG1
        coef[1]=adjw1;
        coef[2]=coef0[2]*coef[1];
        coef[3]=coef0[3]*coef[1];
        coef[4]=coef0[4]*coef[1];
        wmax = 0.0;
        for (k=0; k<ZDIM; k++) {
	  zkm = k*ZRESO*cos_antang[j] - sfcht[j];
          w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	            coef[3]*pow(zkm,2.) + coef[4]*zkm;
          if(zkm < 0.0) w[k][j]=0.0;
          if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
          if(wmax < w[k][j]) wmax=w[k][j];
        }
        printf("# 2nd guess: a=%10.6f, b=%10.6f, c=%10.6f, d=%10.6f  |  wmax=%6.2f\n",
                coef[1],coef[2],coef[3],coef[4],wmax);
#endif



/*=================================================================*/
/* ------ initialize for iteration ------ */
        difLH=99.9;   
        difLH0=99.9;
        difLH1=99.9;
        adjadv=0.0;    /* adjustment of horiz advection of qp  */
        it2=0;

/* ------ iteration loop ------ */
        for(it=0; it<ITMAX; it++){
          if (fabs(difLH) > LH_THRES){
	    if(it==0) adjw = adjw0;    /* 1st guess estimated on the above */
            if(it==1) adjw = adjw1;    /* 2nd guess estimated on the above */

   /* ---- calculate the w-profile ---- */
          coef[1]=adjw;
          coef[2]=coef0[2]*coef[1];
          coef[3]=coef0[3]*coef[1];
          coef[4]=coef0[4]*coef[1];
          wmax=0.0;
          for (k=0; k<ZDIM; k++) {
 	     zkm = k*ZRESO*cos_antang[j] - sfcht[j];
             w[k][j] = coef[1]*pow(zkm,4.) + coef[2]*pow(zkm,3.) + 
	               coef[3]*pow(zkm,2.) + coef[4]*zkm;
             if(zkm < 0.0) w[k][j]=0.0;
             if(zkm > cldt[j]-sfcht[j])  w[k][j]=0.0;
             if(wmax < w[k][j]) wmax=w[k][j];
          }

#ifdef DEBUG
          if(it==0 && wmax>100.0){
            printf("####### abnormal w-prof at j=%2d wtype=%1d ",
                   j,wtype);
            printf("sfcR=%7.3f : adjw=%f wmax=%f adjadv=%4.2f  ",
	           sfcr[j],adjw,wmax,adjadv);
            printf("w=0 at %5.2f, %5.2f,%5.2f,%5.2f\n",
                   zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
          }
#endif
#ifdef DEBUG2
	  if(it<30){
            if(it==0) printf("\n");
            printf("#======== profiles at it=%2d(%2d): j=%2d wtype=%1d ",
	           it,it2,j,wtype);
            printf("sfcR=%7.3f : adjw=%f wmax=%f adjadv=%4.2f ==========\n",
	           sfcr[j],adjw,wmax,adjadv);
          }
#endif

   /* ---- calculate the LH-profile -----------------------------------------
         Fqp = advqph + advqp +srcqp
            advqph = u*dqp/dx + v*dqp/dy  (unknown term --> for adjustment)
            advqp  = w*dqp/dz             (vertical advection)
            srcqp  = (1/rho)*d(rho*vt*qp)/dz  
           (srcqp is also expressed as srcqp = - (1/rho)*dR/dz)
         Fqp>0 && w>0    : LH = -(Lv/Cp) * w * (dqvs/dz) * 3600 [K/hr]
         Fqp>0 && w<0    : LH = 0
         Fqp<0 && w+Vt>0 : LH = 0
         Fqp<0 && w<0    : LH = Fqp * 3600                      [K/hr]
      --------------------------------------------------------------------- */
          /* get Fqp and LH-profile [K/hr] */
          for(k=1; k<ZDIM-1; k++){
             if (qp[k+1][j]>0.0 && qp[k-1][j]>0.0){
	        /* vertical advection of qp */
	        advqp=w[k][j]*(qp[k+1][j]-qp[k-1][j])/(dzm*2.0);
                /* source of qp */
                srcqp=(1.0/rho[k])*(rho[k+1]*vt[k+1][j]*qp[k+1][j]
                                   -rho[k-1]*vt[k-1][j]*qp[k-1][j])/(dzm*2.0);
                /* assume unkwon horizontal advection of qp for adjustment */
                advqph = adjadv*(advqp+srcqp);     /* adjadv: -0.1 to -1.0 */
                xvt0 = (vt0[k+1][j]+vt0[k-1][j])*0.5;

                /* ---<old version for snow region>---
                if(xvt0>SNOW_THRES && adjadv==0.0){ 
                  advqph *= -0.7*(advqp+srcqp);
                }
                -------------------------------------- */

                /* large advqph assumed in anvil region */
                if(wtype==3 && adjadv> -0.7){ 
                  advqph *= -0.7*(advqp+srcqp);
                }
                /* w+vt0>0 & fqp<0 means positive advection */
                pmotion = w[k][j]+vt[k][j];
                if(pmotion>0 && (advqp+srcqp)<0){
                  advqph = -1.01*(advqp+srcqp);
                }
                /* w<0 & fqp>0 means negative advection */
                if(w[k][j]<0 && (advqp+srcqp)>0){
                  advqph = -1.01*(advqp+srcqp);
                }

                fqp[k][j]=advqph + advqp + srcqp;

                /* LH-profile */
                dqvsdz=(qvs[k+1]-qvs[k-1])/(dzm*2.0);

                if(fqp[k][j] >= 0.0){   /* saturation */
                  xevacon=-1.0*w[k][j]*dqvsdz;
                }else{                  /* un-saturation */
		  xevacon=fqp[k][j];
                }

                if(xvt0>SNOW_THRES){  /* snow region */
                   lh[k][j]=((clv+clf)/cp)*xevacon*3600.0;   /* [K/hr] */
                }else{
                   lh[k][j]=(clv/cp)*xevacon*3600.0;
                }
                /* lh[k][j]=(clv/cp)*xevacon*3600.0; /* not consider ice phase */

#ifdef DEBUG2
	      if(it<30){
                printf("k=%2d w=%6.1f vt=%6.1f qp=%10.2e dqp/dz=%10.2e ",
		       k,w[k][j],vt[k][j],qp[k][j],(qp[k+1][j]-qp[k-1][j])/(dzm*2.0));
                printf("advqph=%10.2e advqp=%10.2e srcqp=%10.2e | fqp=%10.2e LH=%8.2f\n",
                        advqph,advqp,srcqp,fqp[k][j],lh[k][j]);
              }
#endif

             }
          }
          /* fill the lowest (echo bottom) fqp and lh */
          lowest_flag=0;
          for(k=0; k<ZDIM-2; k++){
             if(lowest_flag==0 && qp[k][j]>UNDEF2 && 
                fqp[k+1][j]>UNDEF2 && fqp[k+2][j]>UNDEF2 &&
                lh[k+1][j]>UNDEF2 && lh[k+2][j]>UNDEF2){ 
	           fqp[k][j] = 2.0*fqp[k+1][j] - fqp[k+2][j];
	           lh[k][j] = 2.0*lh[k+1][j] - lh[k+2][j];
                   lowest_flag=1;
             }
          }

   /* ---- comparison between integ-LH and srfR-LH ------------------
         Integ[rho * LH/3600]dz = (Lv/Cp) * Rsfc/3600  : Rsfc [mm/h]
                                                       : LH [K/hr]
      --------------------------------------------------------------- */
          /* vertical integration of LH */
          LHinteg=0.0;  LHintegN=0.0;
          for(k=0; k<ZDIM; k++){
	     if(lh[k][j]>UNDEF2){
                 LHinteg+=rho[k]*(lh[k][j]/3600.)*dzm;
                 LHintegN +=1.0;
             }
	     if(lh[k][j]>UNDEF2){
	        if(lh[k][j]<=(UNDEF+0.1)){   /* never occur */
		   printf("####### Abnormal UNDEF value: lh=%f\n",
			  lh[k][j]);
                }
             }
          }
	  /* compare LHinteg and sfcR */
          if(sfcr[j]>minsfcR){
             LHsfcR = (clv/cp)*sfcr[j]/3600.;
          }else{
             LHsfcR = 0.0;
          }
          if(LHintegN > 2.0){
             difLH=LHinteg-LHsfcR;
          }else{
	     difLH=0.0;
          }

   /* ---- adjustment of the w-profile coefficients (bisection method) ---- */
          /* check the first guess */
          if(it==0 && difLH > LH_THRES){  /* usually difLH <= 0 */    
             printf("### 1st guess ERROR at j=%2d  wtype=%1d  sfcR=%5.1f  difLH=%f",
                     j,wtype,sfcr[j],difLH);
             printf(" ----- stop the iteration\n");
	     printf("                       trop=%5.2f cldt=%5.2f melt=%5.2f sfcHT=%5.2f\n",
                     zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
             flag_iter[j]=2.0;
             difLH=0.0;   /* means stop the iteration */
          }
          if(it==0)  difLH0 = difLH;
          if(it==1)  difLH1 = difLH;
          if(it > 0){
             if((difLH0*difLH1)<0.0){
	        /* bisection method */
	        it2++; 
                if((difLH*difLH0)>0.0){  /* replace the edge to midium */
	           difLH0 = difLH;
	           adjw0  = adjw;
	        }else{
                   difLH1 = difLH;
                   adjw1  = adjw;
                }
	        adjw = adjw0 + (adjw1-adjw0)*0.5;    /* set adjw on the bisection */
             }else{
                /* failed to get the effective 2nd guess (now still it2=0) */
	        if(it2>0){     /* check the bisection (maybe no need) */
		   printf("####### FAILED bisection nethod: it=%d(%d) ",it,it2);
                   printf(" difLH=%f difLH0=%f difLH1=%f \n",difLH,difLH0,difLH1);
                   flag_iter[j]=4.0;
                   difLH=0.0;   /* means stop the iteration */
                } 

	        difLH1 = difLH;
                if(0<it && it<10){
                   if(sfcr[j]<minsfcR){
                      guessR = minsfcR * (it+1)*multipR;
                   }else{
                      guessR = sfcr[j] * (it+1)*multipR;
                   }
	           if(guessR>300.0) guessR = 300.0;
                   guessLH = (clv/cp)*guessR/3600.; 
                   adjw=guessLH/LHintegS;
                }
                if(10<=it && it<ITMAX-1){
                   if(sfcr[j]<minsfcR){
                      guessR = minsfcR * 10.0*multipR;
                   }else{
                      guessR = sfcr[j] * 10.0*multipR;
                   }
	           if(guessR>300.0) guessR = 300.0;
                   guessLH = (clv/cp)*guessR/3600.; 
                   adjw=guessLH/LHintegS;
                   if(10<=it && it<=20) adjadv=(it-9)*(-0.1);
                   if(20<it){    /* if the 2nd guess has not gotten. 1st guess is used */
                     printf("### 2nd guess ERROR at j=%2d wtype=%1d sfcR=%5.1f difLH=%f",
                            j,wtype,sfcr[j],difLH);
                     printf(" ----- stop the iteration %2d(%2d) wmax=%6.2f adjadv=%4.1f\n",
                            it,it2,wmax,adjadv);
	             printf("                       ");
	             printf("trop=%5.2f cldt=%5.2f melt=%5.2f sfcHT=%5.2f ---> 1st guess is used\n",
                            zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);

                     guessR  = sfcr[j];
                     if(sfcr[j]<minsfcR) guessR = minsfcR;
                     guessLH = (clv/cp)*guessR/3600.; 
                     adjw=guessLH/LHintegS;
                     adjadv=0.0;
                     flag_iter[j]=3.0;
                     it=ITMAX-2;  /* to calc the 1st guess results at the last iteration */
                   }
                }
	    }  /* end of else if(difLH0*difLH1<0) */

            /* case of failure in the iteration */
            if(it==ITMAX-1){
               printf("=== ITERATION FAILED at j=%2d wtype=%1d sfcR=%6.2f: ",
                       j,wtype,sfcr[j]);
               printf("Iter=%2d(%2d) wmax=%6.2f adjw=%7.4f adjadv=%4.1f ",
		       it,it2,wmax,adjw,adjadv);
               printf("w=0 at %5.2f, %5.2f,%5.2f,%5.2f\n",
                       zt+sfcht[j],zc+sfcht[j],zm+sfcht[j],sfcht[j]);
               printf("                        difLH=%6.2f difLH0=%6.2f difLH1=%6.2f  ",
		       difLH,difLH0,difLH1);
               printf("LHsfcR=%7.2f vs LHinteg=%7.2f (LHintegN=%2.0f)\n",
	               LHsfcR,LHinteg,LHintegN);
               flag_iter[j]=1.0;
            }            

          }  /* end of if(it>1) */

#ifdef DEBUG1
       printf("= Iter=%2d(%2d) j=%2d wtype=%1d:  difLH=%8.3f difLH0=%8.3f difLH1=%8.3f ",
	       it,it2,j,wtype,difLH,difLH0,difLH1);
       printf("adjw=%f adjw0=%f adjw1=%f wmax=%f adjadv=%f\n",
	       adjw,adjw0,adjw1,wmax,adjadv);
       printf("                             LHsfcR=%7.3f vs LHinteg=%8.3f (LHintegN=%3.0f)\n",
	       LHsfcR,LHinteg,LHintegN);
#endif
#ifdef DEBUG2
	  if(it<30){
            printf("========= difLH=%6.2f difLH0=%6.2f difLH1=%6.2f  ",
                    difLH,difLH0,difLH1);
            printf("adjw=%8.6f adjw0=%8.6f adjw1=%8.6f wmax=%5.2f adjadv=%4.1f\n",
	            adjw,adjw0,adjw1,wmax,adjadv);
          }
#endif

          /* store the previous iteration values */
          iterN[j]=it;

          }    /* if(fabs(difLH) > LH_THRES) */

	}  /* end of iteration loop for(it=0; it<ITMAX, it++) */
/*=================================================================*/

      }    /* end of if(wtype>0) */
#ifdef DEBUG1
      if(wtype>0){
         printf("=== CONVERGENCED at Iter Num=%2d(%2d) at j=%2d ",iterN[j],it2,j);
         printf("  wtype=%d  sfcR=%6.2f",wtype,sfcr[j]);
         printf("  difLH=%8.3f difLH0=%8.3f difLH1=%8.3f integN=%3.0f\n",
                   difLH,difLH0,difLH1,LHintegN);
      }   
#endif

      if(iterN_max < iterN[j]) iterN_max = iterN[j];

      }  /* end of angle_bin loop */

      return iterN_max;
}
